(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    654599,      16553]*)
(*NotebookOutlinePosition[    655805,      16592]*)
(*  CellTagsIndexPosition[    655761,      16588]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Chapter 7
Continuous Systems with Cubic Nonlinearities\
\>", "Title",
  TextAlignment->Left,
  TextJustification->0],

Cell[TextData[{
  "Elastic systems, such as beams, plates, and shells, are usually modeled by \
partial-differential equations with specified boundary conditions. For small \
oscillations, the responses of such deformable bodies, ",
  StyleBox["continuous",
    FontColor->RGBColor[0, 0, 1]],
  " or ",
  StyleBox["distributed-parameter systems",
    FontColor->RGBColor[0, 0, 1]],
  ", can be adequately described by linear equations and boundary conditions. \
However, as the amplitude of oscillations increases, nonlinear effects in the \
governing equations, boundary conditions, or both come into play. The sources \
of nonlinearities may be ",
  StyleBox["geometric",
    FontColor->RGBColor[0, 0, 1]],
  ", ",
  StyleBox["inertial",
    FontColor->RGBColor[0, 0, 1]],
  ", ",
  StyleBox["material",
    FontColor->RGBColor[0, 0, 1]],
  ", or",
  StyleBox[" damping",
    FontColor->RGBColor[0, 0, 1]],
  " in nature."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  StyleBox["Geometric",
    FontColor->RGBColor[0, 0, 1]],
  " ",
  StyleBox["nonlinearities",
    FontColor->RGBColor[0, 0, 1]],
  " may be due to one or more of the following: nonlinear relationships among \
the strains and the displacements, large rotations, free surfaces in fluids, \
time-dependent constraints, mid-plane stretching, and large curvatures. \
Nonlinear stretching of the midplane of a deformable body accompanies its \
transverse vibrations if it is supported in such a way as to restrict the \
movement of its ends and / or edges. If large-amplitude vibrations are \
accompanied by large changes in the curvature, it is necessary to employ a \
nonlinear relationship between the curvature and the displacement. ",
  StyleBox["Inertial nonlinearities",
    FontColor->RGBColor[0, 0, 1]],
  " are caused by concentrated and / or distributed masses, convective \
accelerations, and Corriolis and centripetal accelerations. ",
  StyleBox["Material nonlinearities",
    FontColor->RGBColor[0, 0, 1]],
  " occur whenever the constitutive relations are nonlinear, such as the case \
when the stresses are nonlinear functions of the strains; the resistive, \
inductive, and capacitive circuit elements are nonlinear; and the feedback \
control forces and moments in servomechanisms are nonlinear. ",
  StyleBox["Damping nonlinearities ",
    FontColor->RGBColor[0, 0, 1]],
  "occur due to form drag and hysteresis."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Since exact solutions are, in general, not available for determining the \
dynamic responses of nonlinear continuous systems to external or parametric \
excitations, recourse has been made to approximate analyses by using either \
purely numerical techniques, or purely analytical techniques, or a \
combination of numerical and analytical techniques. Application of purely \
numerical techniques to such problems may be costly in terms of computation \
time and may not reveal some of their intricate and complicated responses. \
With purely analytical methods, it may be difficult to treat systems with \
inhomogeneities or complicated geometries. With a combination of numerical \
and analytical techniques, one can determine some of the complicated \
responses of systems with inhomogeneities and complicated shapes.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
The numerical-analytic approaches can be divided into two groups: \
discretization and direct methods.  In the discretization methods, one \
postulates the solution in the form\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "w[x,t]=",
  Cell[BoxData[
      \(TraditionalForm\`\[Sum]\_\(m = 1\)\%M\)]],
  Cell[BoxData[
      \(TraditionalForm\`\(\(\[Phi]\_m\)(x)\)\ \(\(q\_m\)(t)\)\)]],
  "   "
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  StyleBox["M",
    FontSlant->"Italic"],
  " is a finite integer. Then, one assumes the spatial functions ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\_m\)(x)\)]],
  ", space discretization, or the temporal functions ",
  Cell[BoxData[
      \(TraditionalForm\`\(q\_m\)(t)\)]],
  ", time discretization. With this discretization, the ",
  Cell[BoxData[
      \(TraditionalForm\`\(q\_m\)(t)\)]],
  " are usually taken to be harmonic and the method of harmonic balance is \
used to obtain a set of nonlinear boundary-value problems for the ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\_m\)(x)\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "With space discretization, the ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\_m\)(x)\)]],
  " (and thus the spatial dependence) are assumed a priori. If the boundary \
conditions are homogeneous, the ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\_m\)(x)\)]],
  " are usually taken to be the eigenfunctions of the linearized problem. The \
method of weighted residuals or variational principles can then be used to \
determine a set of ordinary-differential equations governing the ",
  Cell[BoxData[
      \(TraditionalForm\`\(q\_m\)(t)\)]],
  ". The obtained set of ordinary-differential equations can be studied using \
any of a great number of methods developed for discrete systems."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
The most common implementation of weighted residuals is the Galerkin method. \
The great majority of recent studies of forced vibrations assume that the \
response can be expressed in terms of only the linear modes that are directly \
or indirectly excited (Nayfeh and Mook, 1979). For example, if a system is \
driven near the natural frequency of a linear mode and that mode is not \
involved in an internal resonance with other modes, the response is assumed \
to consist of only that mode. Such an approach is usually referred to as a \
single-mode approximation.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
In the direct approach, a reduction method, such as the method of multiple \
scales, is applied directly to the governing partial-differential equations \
and associated boundary conditions and no assumptions are made a priori \
regarding the spatial or temporal dependence of the response. The principal \
advantage of this approach is in the treatment of the boundary conditions at \
higher orders. This approach has been used recently by Nayfeh and Nayfeh \
(1979), Nayfeh (1975, 1996), Nayfeh and Asfar (1986), Nayfeh and Bouguerra \
(1990), Pai and Nayfeh (1990), Raouf and Nayfeh (1990), Nayfeh, Nayfeh, and \
Mook (1992), Nayfeh and Nayfeh (1993, 1995), Nayfeh, Nayfeh, and Pakdemirli \
(1995), Pakdemirli, Nayfeh, and Nayfeh (1995), Chin and Nayfeh (1996), Nayfeh \
and Lacarbonara (1997, 1998), Lacarbonara, Nayfeh, and Kreider (1998), \
Nayfeh, Lacarbonara, and Chin (1999), and Rega, et al. (1999).\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Some of the aforementioned studies show that the discretization and direct \
approaches yield the same results for systems with cubic nonlinearities \
provided that first-order approximations are sought. For systems with \
quadratic and cubic nonlinearities, the discretization approach might produce \
quantitative, and in some case qualitative, errors unless many modes are \
included in the discretized model. In other words, one has to include as many \
terms in the discretized model as needed for convergence. Since we are \
dealing with systems with cubic nonlinearities in this chapter, we will \
obtain first-order results using both the discretization and direct \
approaches.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["Preliminaries", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Off[General::spell1, Integrate::gener]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Needs["\<Utilities`Notation`\>"]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To use the method of multiple scales, we introduce different time scales ",
  
  Cell[BoxData[
      \(TraditionalForm\`\(\(T\_0 = \ t\ , \)\ \)\)]],
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  "= \[Epsilon]",
  StyleBox[" t",
    FontSlant->"Italic"],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  "= ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^2\)]],
  StyleBox["t",
    FontSlant->"Italic"],
  ", symbolize them as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_0\),
          NotationBoxTag,
          Editable->True], "]"}], ";", 
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_1\),
          NotationBoxTag,
          Editable->True], "]"}], ";", 
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_2\),
          NotationBoxTag,
          Editable->True], "]"}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and form a list of them as follows:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(timeScales = {T\_0, T\_1, T\_2}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In terms of the time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_n\)]],
  ", the time derivatives become"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(dt[0]\)[expr_] := expr; 
    \(dt[1]\)[expr_] := 
      Sum[\[Epsilon]\^i\ D[expr, timeScales[\([i + 1]\)]], {i, 0, 1}]; 
    \(dt[2]\)[expr_] := 
      \((\(dt[1]\)[\(dt[1]\)[expr]] // Expand)\) /. 
        \[Epsilon]\^\(i_ /; i > 1\) -> 0; \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In the course of the analysis, we need the complex conjugates of ",
  Cell[BoxData[
      \(TraditionalForm\`A\)]],
  " and \[CapitalGamma]. We define them using the following rule:"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(conjugateRule = {A -> A\&_, A\&_ -> A, 
        \[CapitalGamma] -> \[CapitalGamma]\&_, 
        \[CapitalGamma]\&_ -> \[CapitalGamma], 
        Complex[0, n_] -> Complex[0, \(-n\)]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To manipulate some complicated integrals without ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " being choked, we define the following rules:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(intRule1 = {int[fun_, arg2__] :> int[Expand[fun], arg2]}; \n
    intRule2 = {int[a_ + b_, arg2__] :> int[a, arg2] + int[b, arg2], 
        int[\[Epsilon]\^n_. \ fun_, arg2__] :> \[Epsilon]\^n\ int[fun, arg2], 
        int[a_\ fun_, a1_, b1___] :> 
          a\ int[fun, a1, b1] /; FreeQ[a, First[a1]], 
        int[int[a1_, a2_]\ fun_. , a3__] :> 
          int[a1, a2]\ int[fun, a3] /; FreeQ[Rest[a2], First[a2]]}; \)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To represent some of the expressions in a more concise way, we introduce the \
following display rule:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(displayRule = {
        \(\(Derivative[a_, b__]\)[w_\_i_]\)[x, __] :> 
          SequenceForm[
            If[arg1 = 
                Times\ @@\ MapIndexed[D\_\(#2[\([1]\)] - 1\)\%#1\ &, {b}]; 
              arg1 =!= 1, arg1, "\<\>"], 
            w\_i\^SequenceForm\ @@\ Table["\<'\>", {a}]], 
        \(\(Derivative[a__]\)[A\_i_]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\_i], \(\(Derivative[a__]\)[A\&_\_i_]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\&_\_i], w_\_i_[x, __] -> w\_i, A\_i_[__] -> A\_i, 
        A\&_\_i_[__] -> A\&_\_i, 
        Exp[a_.  + b_. \ Complex[0, m_]\ T\_0 + 
              c_. \ Complex[0, n_]\ T\_0] :> 
          Exp[a + \((m*b + n*c)\) I\ T\_0], int -> Integrate}; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.1 Solvability Conditions and the Concept of Adjoint", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In directly attacking continuous systems, one often encounters \
nonhomogeneous boundary-value problems whose homogeneous parts have \
nontrivial solutions. Consequently, the nonhomogeneous problems have \
solutions only if ",
  StyleBox["solvability",
    FontColor->RGBColor[0, 0, 1]],
  " or ",
  StyleBox["consistency conditions",
    FontColor->RGBColor[0, 0, 1]],
  " are satisfied (Nayfeh, 1981). In this section, we describe how to \
determine such solvability conditions."
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell["We start the discussion with the simple problem", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(eq1 = 
      \(y\^\[DoublePrime]\)[x] + \[Pi]\^2\ y[x] == \[Pi]\ Sin[\[Pi]\ x]; \n
    bc1 = {y[0] == \[Beta]\_1, y[1] == \[Beta]\_2}; \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
The corresponding homogeneous problem has the nontrivial solution\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(DSolve[{eq1[\([1]\)] == 0}~Join~bc1 /. \[Beta]\_i_ -> 0, y[x], x]\)[
      \([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({y[x] \[Rule] \(-C[1]\)\ Sin[\[Pi]\ x]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Hence, the nonhomogeneous problem will not have a solution unless a \
solvability condition is satisfied. To determine this solvability condition, \
we use two approaches.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In the first approach, we find the general solution of ",
  StyleBox["eq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " using the function ",
  StyleBox["DSolve",
    FontWeight->"Bold"],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(yRule = \(DSolve[eq1, y[x], x]\)[\([1]\)] // Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({y[x] \[Rule] 
        \((\(-\(x\/2\)\) + C[2])\)\ Cos[\[Pi]\ x] - C[1]\ Sin[\[Pi]\ x]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[1] and ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[2] are arbitrary constants. Imposing the boundary conditions ",
  StyleBox["bc1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(bc2 = bc1 /. Flatten[yRule /. {{x -> 0}, {x -> 1}}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({C[2] == \[Beta]\_1, 1\/2 - C[2] == \[Beta]\_2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["These equations are inconsistent unless", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Equal\ @@\ \(Plus\ @@\ \((\(List\ @@\ #\ &\)\ /@\ bc2)\)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/2 == \[Beta]\_1 + \[Beta]\_2\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which is the desired solvability condition.", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In the second approach, instead of determining the general solution of the \
homogeneous differential equations and then enforcing the boundary conditions \
to determine the solvability conditions, we use the concept of ",
  StyleBox["adjoint",
    FontColor->RGBColor[0, 0, 1]],
  " as described next. This approach is attractive if one is not interested \
in determining the solution of the nonhomogeneous problem but interested only \
in determining the solvability conditions, as is the case in many \
applications of perturbation methods."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine the adjoint and then the solvability conditions, we need to \
perform integration by parts of some products. To accomplish this with ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", we define a function named ",
  StyleBox["intByParts",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(intByParts[expr_ /; Head[expr] =!= Equal, depVar_] := \n\ \ \ 
      Module[{intRule1, intRule2, intRule3}, \n\t\t\ \ 
        intRule1 = f_[terms_, x_] :> int[Expand[terms], x]; \n\ \ \ \ \ \ 
        intRule2 = int[term1_ + term2_, x_] :> int[term1, x] + int[term2, x]; 
        \n\ \ \ \ \ \ 
        intRule3 = 
          c1_. \ int[term1_, x_] + c2_. \ int[term2_, x_] :> 
            int[c1\ term1 + c2\ term2, x]; \n\ \ \ \ \ \ 
        IBP[u_, v_, n_ /; n >= 1, range : {x_, a_, b_}] := 
          \((u\ \(\(Derivative[n - 1]\)[v]\)[x] /. x -> b)\) - 
            \((u\ \(\(Derivative[n - 1]\)[v]\)[x] /. x -> a)\) - 
            IBP[D[u, x], v, n - 1, range]; \n\ \ \ \ \ \ 
        IBP[u_, v_, 0, range : {x_, a_, b_}] := int[u\ v[x], range]; \n\t\t\n
        \ \ \ \ \ \ 
        \(\(\(\(expr /. intRule1\) //. intRule2\) /. 
              int[u_\ \(\(Derivative[n_]\)[depVar]\)[x_], arg2_] :> 
                IBP[u, depVar, n, arg2]\) //. intRule3\) /. int -> Integrate\n
        \ \ ]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine the solvability condition of ",
  StyleBox["eq1 ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "and ",
  StyleBox["bc1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "with this approach, we multiply ",
  StyleBox["eq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " by ",
  Cell[BoxData[
      \(TraditionalForm\`u[x]\)]],
  ", integrate the result by parts from ",
  Cell[BoxData[
      \(TraditionalForm\`x = 0\)]],
  " to ",
  Cell[BoxData[
      \(TraditionalForm\`x = 1\)]],
  ", and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq1a = \(intByParts[int[u[x]\ #, {x, 0, 1}], y]\ &\)\ /@\ eq1\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(\[Integral]\_0\%1\), 
          RowBox[{
            RowBox[{"(", 
              RowBox[{\(\[Pi]\^2\ u[x]\ y[x]\), "+", 
                RowBox[{\(y[x]\), " ", 
                  RowBox[{
                    SuperscriptBox["u", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}]}], ")"}], 
            \(\[DifferentialD]x\)}]}], "+", 
        RowBox[{\(y[0]\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{\(y[1]\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}]}], "-", 
        RowBox[{\(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["y", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{\(u[1]\), " ", 
          RowBox[{
            SuperscriptBox["y", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}]}]}], "==", 
      \(\[Pi]\ \(\[Integral]\_0\%1\( Sin[\[Pi]\ x]\ u[x]\) 
            \[DifferentialD]x\)\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint, we set the coefficient of ",
  Cell[BoxData[
      \(TraditionalForm\`y[x]\)]],
  " in the integrand on the left-hand side of ",
  StyleBox["eq1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(adjointEq1 = 
      \(Cases[eq1a[\([1]\)], Integrate[expr_, _] :> Coefficient[expr, y[x]]]
            \)[\([1]\)] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(\[Pi]\^2\ u[x]\), "+", 
        RowBox[{
          SuperscriptBox["u", "\[DoublePrime]",
            MultilineFunction->None], "[", "x", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint boundary conditions, we consider the homogeneous \
problem (i.e., ",
  Cell[BoxData[
      \(TraditionalForm\`f[x] = 0\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`\[Beta]\_1 = 0\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Beta]\_2 = 0\)]],
  "), use the ",
  StyleBox["adjointEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and obtain from ",
  StyleBox["eq1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(bc1a = 
      \(eq1a /. Integrate -> \((0\ &)\)\) /. 
        \((\(bc1 /. Equal -> Rule\) /. \[Beta]\_i_ -> 0)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(-u[0]\), " ", 
          RowBox[{
            SuperscriptBox["y", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{\(u[1]\), " ", 
          RowBox[{
            SuperscriptBox["y", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint boundary conditions, we set each of the \
coefficients of ",
  Cell[BoxData[
      \(TraditionalForm\`\(y\^\[Prime]\)[0]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\(y\^\[Prime]\)[1]\)]],
  " in ",
  StyleBox["bc1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"adjointBC1", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"Solve", "[", 
            RowBox[{
              RowBox[{
                RowBox[{"Coefficient", "[", 
                  RowBox[{\(bc1a[\([1]\)]\), ",", 
                    RowBox[{"{", 
                      RowBox[{
                        RowBox[{
                          SuperscriptBox["y", "\[Prime]",
                            MultilineFunction->None], "[", "0", "]"}], ",", 
                        RowBox[{
                          SuperscriptBox["y", "\[Prime]",
                            MultilineFunction->None], "[", "1", "]"}]}], 
                      "}"}]}], "]"}], "==", "0"}], ",", \({u[0], u[1]}\)}], 
            "]"}], "[", \([1]\), "]"}], "/.", \(Rule -> Equal\)}]}]], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u[0] == 0, u[1] == 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Therefore, the adjoint ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " is defined by the adjoint system consisting of ",
  StyleBox["adjointEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["adjointBC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Since they are the same as the homogeneous parts of ",
  StyleBox["eq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["bc1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", the problem is said to be self-adjoint. Hence, ",
  Cell[BoxData[
      \(TraditionalForm\`u[x] = Sin[x]\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(adjoint = {u -> \ Function[x, \ Sin[\[Pi]\ x]]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u \[Rule] Function[x, Sin[\[Pi]\ x]]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Once the adjoint problem has been defined, we return to the nonhomogeneous \
problem to determine the solvability condition. Substituting for the adjoint \
and the boundary conditions ",
  StyleBox["bc1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "into ",
  StyleBox["eq1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields the solvability condition"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SolvCond1 = 
      eq1a[\([2]\)] == 
        \((\(\(eq1a[\([1]\)] /. Integrate -> \((0\ &)\)\) /. 
              \((bc1 /. Equal -> Rule)\)\) /. 
            \((adjointBC1 /. Equal -> Rule)\))\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
    \(\[Pi]\ \(\[Integral]\_0\%1\( Sin[\[Pi]\ x]\ u[x]\) 
            \[DifferentialD]x\)\), "==", 
      RowBox[{
        RowBox[{\(\[Beta]\_1\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{\(\[Beta]\_2\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(SolvCondf = SolvCond1 /. adjoint\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Pi]\/2 == \[Pi]\ \[Beta]\_1 + \[Pi]\ \[Beta]\_2\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which is the same as the solvability condition obtained with the other \
method.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["7.1.1 Hinged-Clamped Beam", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
In treating a hinged-clamped beam, we need to determine the adjoint of the \
boundary-value problem\
\>", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"eq1a", "=", 
        RowBox[{
          RowBox[{\(\(-\[Omega]\^2\)\ \[Phi][x]\), "-", 
            RowBox[{"2", " ", "P", " ", 
              RowBox[{
                SuperscriptBox["\[Phi]", "\[DoublePrime]",
                  MultilineFunction->None], "[", "x", "]"}]}], "+", 
            RowBox[{
              SuperscriptBox["\[Phi]", 
                TagBox[\((4)\),
                  Derivative],
                MultilineFunction->None], "[", "x", "]"}]}], "==", 
          \(f[x]\)}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"bc1a", "=", 
        RowBox[{"{", 
          RowBox[{\(\[Phi][0] == 0\), ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox["\[Phi]", "\[DoublePrime]",
                  MultilineFunction->None], "[", "0", "]"}], "==", "0"}], 
            ",", \(\[Phi][1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox["\[Phi]", "\[Prime]",
                  MultilineFunction->None], "[", "1", "]"}], "==", "0"}]}], 
          "}"}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Multiplying ",
  StyleBox["eq1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " by ",
  Cell[BoxData[
      \(TraditionalForm\`u[x]\)]],
  " and integrating the result by parts from ",
  Cell[BoxData[
      \(TraditionalForm\`x = 0\)]],
  " to ",
  Cell[BoxData[
      \(TraditionalForm\`x = 1\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq1b = \(intByParts[int[u[x]\ #, {x, 0, 1}], \[Phi]]\ &\)\ /@\ eq1a\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(\[Integral]\_0\%1\), 
          RowBox[{
            RowBox[{"(", 
              RowBox[{\(\(-\[Omega]\^2\)\ u[x]\ \[Phi][x]\), "-", 
                RowBox[{"2", " ", "P", " ", \(\[Phi][x]\), " ", 
                  RowBox[{
                    SuperscriptBox["u", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], "+", 
                RowBox[{\(\[Phi][x]\), " ", 
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((4)\),
                        Derivative],
                      MultilineFunction->None], "[", "x", "]"}]}]}], ")"}], 
            \(\[DifferentialD]x\)}]}], "-", 
        RowBox[{"2", " ", "P", " ", \(\[Phi][0]\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{"2", " ", "P", " ", \(\[Phi][1]\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}]}], "+", 
        RowBox[{"2", " ", "P", " ", \(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{"2", " ", "P", " ", \(u[1]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}]}], "-", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}]}], "+", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}]}], "+", 
        RowBox[{\(\[Phi][0]\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{\(\[Phi][1]\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}]}], "-", 
        RowBox[{\(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{\(u[1]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}]}]}], "==", 
      \(\[Integral]\_0\%1\( f[x]\ u[x]\) \[DifferentialD]x\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint equation, we set the coefficient of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi][x]\)]],
  " in the integrand on the left-hand side of ",
  StyleBox["eq1b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(adjointEq1 = 
      \(Cases[eq1b[\([1]\)], 
            Integrate[expr_, _] :> Coefficient[expr, \[Phi][x]]]\)[
          \([1]\)] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(\(-\[Omega]\^2\)\ u[x]\), "-", 
        RowBox[{"2", " ", "P", " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((4)\),
              Derivative],
            MultilineFunction->None], "[", "x", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint boundary conditions, we consider the homogeneous \
problem, use the ",
  StyleBox["adjointEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and obtain from ",
  StyleBox["eq1b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(bc1b = 
      \(eq1b /. Integrate -> \((0\ &)\)\) /. \((bc1a /. Equal -> Rule)\)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"2", " ", "P", " ", \(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}]}], "-", 
        RowBox[{\(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{\(u[1]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint boundary conditions, we set each of the \
coefficients of ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\^\[Prime]\)[0]\ , \(\[Phi]\^\((3)\)\)[0]\ , 
      \(\[Phi]\^\[DoublePrime]\)[1]\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\^\((3)\)\)[1]\)]],
  " in ",
  StyleBox["bc1b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"adjointBC1", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"Solve", "[", 
            RowBox[{
              RowBox[{
                RowBox[{"Coefficient", "[", 
                  RowBox[{\(bc1b[\([1]\)]\), ",", 
                    RowBox[{"{", 
                      RowBox[{
                        RowBox[{
                          SuperscriptBox["\[Phi]", "\[Prime]",
                            MultilineFunction->None], "[", "0", "]"}], ",", 
                        RowBox[{
                          SuperscriptBox["\[Phi]", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "0", "]"}], ",", 
                        RowBox[{
                          SuperscriptBox["\[Phi]", "\[DoublePrime]",
                            MultilineFunction->None], "[", "1", "]"}], ",", 
                        RowBox[{
                          SuperscriptBox["\[Phi]", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "1", "]"}]}], 
                      "}"}]}], "]"}], "==", "0"}], ",", 
              \({u[0], \(u\^\[DoublePrime]\)[0], u[1], \(u\^\[Prime]\)[1]}
                \)}], "]"}], "[", \([1]\), "]"}], "/.", 
        \(Rule -> Equal\)}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}], "==", "0"}], ",", 
        \(u[1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], "==", "0"}], ",", 
        \(u[0] == 0\)}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Therefore, the adjoint ",
  StyleBox["u",
    FontSlant->"Italic"],
  " is defined by the adjoint system consisting of ",
  StyleBox["adjointEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["adjointBC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Since they are the same as the homogeneous parts of ",
  StyleBox["eq1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["bc1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", the problem is said to be self-adjoint."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Once the adjoint problem has been defined, we return to the nonhomogeneous \
problem to determine the solvability condition. Substituting for the adjoint \
and the boundary conditions ",
  StyleBox["bc1a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "into ",
  StyleBox["eq1b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields the solvability condition"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SolvCond1 = 
      eq1b[\([2]\)] == 
        \((\(\(eq1b[\([1]\)] /. Integrate -> \((0\ &)\)\) /. 
              \((bc1a /. Equal -> Rule)\)\) /. 
            \((adjointBC1 /. Equal -> Rule)\))\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Integral]\_0\%1\( f[x]\ u[x]\) \[DifferentialD]x == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.1.2 Cantilever Beam", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
The boundary-value problem for a cantilever beam can be written as\
\>", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"eq2a", "=", 
        RowBox[{
          RowBox[{\(\(-\[Omega]\^2\)\ \[Phi][x]\), "+", 
            RowBox[{
              SuperscriptBox["\[Phi]", 
                TagBox[\((4)\),
                  Derivative],
                MultilineFunction->None], "[", "x", "]"}]}], "==", 
          \(f[x]\)}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"bc2a", "=", 
        RowBox[{"{", 
          RowBox[{
          \(\[Phi][0] == 0\), ",", \(\(\[Phi]\^\[Prime]\)[0] == 0\), ",", 
            \(\(\[Phi]\^\[DoublePrime]\)[1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox["\[Phi]", 
                  TagBox[\((3)\),
                    Derivative],
                  MultilineFunction->None], "[", "1", "]"}], "==", "0"}]}], 
          "}"}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine the adjoint of this problem, we multiply ",
  StyleBox["eq2a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " by ",
  Cell[BoxData[
      \(TraditionalForm\`u[x]\)]],
  ", integrate the result by parts from ",
  Cell[BoxData[
      \(TraditionalForm\`x = 0\)]],
  " to ",
  Cell[BoxData[
      \(TraditionalForm\`x = 1\)]],
  ", and obtain"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq2b = \(intByParts[int[u[x]\ #, {x, 0, 1}], \[Phi]]\ &\)\ /@\ eq2a\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(\[Integral]\_0\%1\), 
          RowBox[{
            RowBox[{"(", 
              RowBox[{\(\(-\[Omega]\^2\)\ u[x]\ \[Phi][x]\), "+", 
                RowBox[{\(\[Phi][x]\), " ", 
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((4)\),
                        Derivative],
                      MultilineFunction->None], "[", "x", "]"}]}]}], ")"}], 
            \(\[DifferentialD]x\)}]}], "-", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}]}], "+", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}]}], "+", 
        RowBox[{\(\[Phi][0]\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{\(\[Phi][1]\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}]}], "-", 
        RowBox[{\(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "0", "]"}]}], "+", 
        RowBox[{\(u[1]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}]}]}], "==", 
      \(\[Integral]\_0\%1\( f[x]\ u[x]\) \[DifferentialD]x\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We set the coefficient of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi][x]\)]],
  " in the integrand on the left-hand side of ",
  StyleBox["eq2b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero and obtain the adjoint equation as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(adjointEq2 = 
      \(Cases[eq2b[\([1]\)], 
            Integrate[expr_, _] :> Coefficient[expr, \[Phi][x]]]\)[
          \([1]\)] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(\(-\[Omega]\^2\)\ u[x]\), "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((4)\),
              Derivative],
            MultilineFunction->None], "[", "x", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint boundary conditions, we consider the homogeneous \
problem, use the ",
  StyleBox["adjointEq2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and obtain from ",
  StyleBox["eq2b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(bc2b = 
      \(eq2b /. Integrate -> \((0\ &)\)\) /. \((bc2a /. Equal -> Rule)\)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Phi]", "\[Prime]",
              MultilineFunction->None], "[", "1", "]"}], " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}]}], "+", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}]}], "-", 
        RowBox[{\(\[Phi][1]\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}]}], "-", 
        RowBox[{\(u[0]\), " ", 
          RowBox[{
            SuperscriptBox["\[Phi]", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "0", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the adjoint boundary conditions, we set each of the \
coefficients of ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\^\[DoublePrime]\)[0]\ , 
      \(\[Phi]\^\((3)\)\)[0]\ , \[Phi][1]\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\^\[Prime]\)[1]\)]],
  " in ",
  StyleBox["bc2b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"adjointBC2", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"Solve", "[", 
            RowBox[{
              RowBox[{
                RowBox[{"Coefficient", "[", 
                  RowBox[{\(bc2b[\([1]\)]\), ",", 
                    RowBox[{"{", 
                      RowBox[{\(\(\[Phi]\^\[DoublePrime]\)[0]\), ",", 
                        RowBox[{
                          SuperscriptBox["\[Phi]", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "0", "]"}], ",", 
                        \(\[Phi][1]\), ",", \(\(\[Phi]\^\[Prime]\)[1]\)}], 
                      "}"}]}], "]"}], "==", "0"}], ",", 
              RowBox[{"{", 
                RowBox[{
                \(u[0]\), ",", \(\(u\^\[Prime]\)[0]\), ",", 
                  \(\(u\^\[DoublePrime]\)[1]\), ",", 
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((3)\),
                        Derivative],
                      MultilineFunction->None], "[", "1", "]"}]}], "}"}]}], 
            "]"}], "[", \([1]\), "]"}], "/.", \(Rule -> Equal\)}]}]], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(u[0] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "0", "]"}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((3)\),
                Derivative],
              MultilineFunction->None], "[", "1", "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Therefore, the adjoint ",
  StyleBox["u",
    FontSlant->"Italic"],
  " is defined by the adjoint system consisting of ",
  StyleBox["adjointEq2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["adjointBC2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Since they are the same as the homogeneous parts of ",
  StyleBox["eq2a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["bc2a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", the problem is said to be self-adjoint."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Once the adjoint problem has been defined, we return to the nonhomogeneous \
problem to determine the solvability condition. Substituting for the adjoint \
and the boundary conditions ",
  StyleBox["bc2a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "into ",
  StyleBox["eq2b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields the solvability condition"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SolvCond2 = 
      eq2b[\([2]\)] == 
        \((\(\(eq2b[\([1]\)] /. Integrate -> \((0\ &)\)\) /. 
              \((bc2a /. Equal -> Rule)\)\) /. 
            \((adjointBC2 /. Equal -> Rule)\))\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Integral]\_0\%1\( f[x]\ u[x]\) \[DifferentialD]x == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.2 Hinged-Clamped Beam", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[TextData[StyleBox["7.2.1 EOM and BC's",
  FontWeight->"Bold"]], "Subsection",
  TextAlignment->Left,
  TextJustification->1,
  Background->None],

Cell[TextData[{
  "We consider the nonlinear planar response of a hinged-clamped uniform \
prismatic beam to a harmonic axial load. The beam is subjected to a static \
axial load and one of its ends is restrained by a linear spring. We assume \
that the curvature and inertia nonlinearities are much smaller than the \
nonlinearity caused by the mid-plane stretching. The transverse deflection ",
  Cell[BoxData[
      \(TraditionalForm\`w[x, t]\)]],
  " of the beam at the position ",
  StyleBox["x",
    FontSlant->"Italic"],
  " and time ",
  StyleBox["t",
    FontSlant->"Italic"],
  " is governed by the nondimensional integral-partial-differential equation \
(Nayfeh and Mook, 1979)"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(EOM = 
      \[PartialD]\_\(t, t\)w[x, t] - 2  P\ \[PartialD]\_\(x, x\)w[x, t] + 
          \[PartialD]\_\(x, x, x, x\)w[x, t] == 
        \(-2\) \[Epsilon]\ \[Mu][x]\ \[PartialD]\_t w[x, t] + 
          4  \[Epsilon]\ F\ \[PartialD]\_\(x, x\)w[x, t]\ 
            Cos[\[CapitalOmega]\ t] + 
          \[Epsilon]\ \[Alpha]\ \[PartialD]\_\(x, x\)w[x, t]\ 
            \(\[Integral]\_0\%1\(\((\[PartialD]\_x w[x, t])\)\^2\) 
                \[DifferentialD]x\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((0, 2)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}], "-", 
        RowBox[{"2", " ", "P", " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((4, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}]}], "==", 
      RowBox[{
        RowBox[{\(-2\), " ", "\[Epsilon]", " ", \(\[Mu][x]\), " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
        "4", " ", "F", " ", "\[Epsilon]", " ", \(Cos[t\ \[CapitalOmega]]\), 
          " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{"\[Alpha]", " ", "\[Epsilon]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SuperscriptBox["w", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}]}]}]], "Output",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and homogeneous boundary conditions", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC = {w[0, t] == 0, \[PartialD]\_\(x, x\)w[x, t] == 0 /. x -> 0, 
        w[1, t] == 0, \[PartialD]\_x w[x, t] == 0 /. x -> 1}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(w[0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, t\), "]"}], "==", "0"}], 
        ",", \(w[1, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
We assume that the static axial load is such that the lowest two natural \
frequencies of the beam are in the ratio of three-to-one; that is, there is a \
three-to-one internal resonance between the second and first modes. Moreover, \
we assume that neither of these two modes is involved in an internal \
resonance with any other mode. We consider three parametric resonances, \
namely, principal parametric resonance of the first mode, principal \
parametric resonance of the second mode, and combination parametric resonance \
of the first and second modes.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.2.2 Direct Attack of the Continuous Problem", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this section, we directly attack the integral-partial-differential \
equation ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and associated boundary conditions ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and seek a first-order uniform expansion of their solution in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(solRule = 
      w -> \((Evaluate[Sum[\[Epsilon]\^j\ w\_j[#1, #2, #3], {j, 0, 1}]]\ &)
          \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(w \[Rule] \((w\_0[#1, #2, #3] + \[Epsilon]\ w\_1[#1, #2, #3]&)\)\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the first independent variable stands for ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and the last two independent variables stand for the two time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ". Substituting the ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", transforming the total time derivatives into partial derivatives in \
terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", expanding the result for small \[Epsilon], discarding terms of order \
higher than \[Epsilon], and using the ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to simplify the expansions of the integrands, we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq722a = 
      \((\((\(\(Join[{EOM}, BC] /. Integrate -> int\) /. {
                        w[x_, t] -> w[x, T\_0, T\_1], 
                        \(\(Derivative[m_, n_]\)[w]\)[x_, t] -> 
                          \(dt[n]\)[D[w[x, T\_0, T\_1], {x, m}]], t -> T\_0}
                        \) /. solRule // ExpandAll)\) //. intRule2 // 
            ExpandAll)\) /. \[Epsilon]\^\(n_ /; n > 1\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq722a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      \(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@\ 
          eq722a // Transpose; \neqEps /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{\(-2\), " ", "P", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "0", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 0], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "0", 
                      
                      InterpretationBox[
                        \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'", "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ w, 0], 
                      SequenceForm[ "'", "'", "'", "'"]]],
                  Editable->False], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "0", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ w, 0], 
                      SequenceForm[ ]]],
                  Editable->False]}], "==", "0"}], ",", 
            \(w\_0[0, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "0", 
                  TagBox[\((2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_0[1, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "0", 
                  TagBox[\((1, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], 
              "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{
                \(-4\), " ", "F", " ", \(Cos[T\_0\ \[CapitalOmega]]\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "0", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 0], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"\[Alpha]", " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "0", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 0], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "0", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 0], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", "P", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "1", 
                      
                      InterpretationBox[
                        \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'", "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ w, 1], 
                      SequenceForm[ "'", "'", "'", "'"]]],
                  Editable->False], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ w, 1], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "0", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ w, 0], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "0", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ w, 0], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(\[Mu][x]\)}]}], "==", 
              "0"}], ",", \(w\_1[0, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "1", 
                  TagBox[\((2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_1[1, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "1", 
                  TagBox[\((1, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Because in the presence of damping, all modes that are not directly \
excited by the forcing or indirectly excited by the internal resonance will \
decay with time (Nayfeh and Mook, 1979), the solution of ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be expressed in terms of the lowest two linear free-vibration modes; \
that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol0 = 
      w\_0 -> Function[{x, T\_0, T\_1}, 
          Sum[A\_i[T\_1]\ \[Phi]\_i[x]\ Exp[I\ \[Omega]\_i\ T\_0] + 
              A\&_\_i[T\_1]\ \[Phi]\_i[x]\ Exp[\(-I\)\ \[Omega]\_i\ T\_0], {
              i, 2}]]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_\(1\ \)\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_2\)]],
  " are the natural frequencies of these modes. For later use, we define the \
list"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(omgList = {\[Omega]\_1, \[Omega]\_2}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "One can easily show that the mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_i[x]\)]],
  " are orthogonal. We assume that these modes are normalized so that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[Cell[BoxData[
    \(\[Integral]\_0\%1 \[Phi]\_i[x]\ \[Phi]\_j[x]\ \[DifferentialD]x -> 
      \[Delta]\_ij\)], "Input"]], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\_ij\)]],
  " is the ",
  "Kronecker",
  " delta function."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol0",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the first-order equation, ",
  StyleBox["eqEps[[2,1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1Eq = 
      \((eqEps[\([1, 1, 1]\)] /. w\_0 -> w\_1)\) == 
        \((\(\((eqEps[\([1, 1, 1]\)] /. w\_0 -> w\_1)\) - 
                  \((\(Subtract\ @@\ eqEps[\([2, 1]\)] /. sol0 // TrigToExp
                        \) // Expand)\) /. intRule1\) //. intRule2 // Expand)
          \); \norder1Eq /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(-2\), " ", "P", " ", 
          RowBox[{"(", 
            InterpretationBox[
              RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                SubsuperscriptBox["w", "1", 
                  InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                    SequenceForm[ "'", "'"],
                    Editable->False]]}],
              SequenceForm[ "", 
                Power[ 
                  Subscript[ w, 1], 
                  SequenceForm[ "'", "'"]]],
              Editable->False], ")"}]}], "+", 
        InterpretationBox[
          RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
            SubsuperscriptBox["w", "1", 
              
              InterpretationBox[
                \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\[InvisibleSpace]\
"'"\),
                SequenceForm[ "'", "'", "'", "'"],
                Editable->False]]}],
          SequenceForm[ "", 
            Power[ 
              Subscript[ w, 1], 
              SequenceForm[ "'", "'", "'", "'"]]],
          Editable->False], "+", 
        InterpretationBox[
          RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
            SubsuperscriptBox["w", "1", 
              InterpretationBox["\[Null]",
                SequenceForm[ ],
                Editable->False]]}],
          SequenceForm[ 
            Power[ 
              Subscript[ D, 0], 2], 
            Power[ 
              Subscript[ w, 1], 
              SequenceForm[ ]]],
          Editable->False]}], "==", 
      RowBox[{
        RowBox[{\(-2\), " ", "I", " ", \(E\^\(I\ T\_0\ \[Omega]\_1\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ A, 1]],
              Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
          \(\[Phi]\_1[x]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\_1\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ 
                  OverBar[ A], 1]],
              Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
          \(\[Phi]\_1[x]\)}], "-", 
        \(2\ I\ E\^\(I\ T\_0\ \[Omega]\_1\)\ A\_1\ \[Omega]\_1\ \[Mu][x]\ 
          \[Phi]\_1[x]\), "+", 
        \(2\ I\ E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\ \[Omega]\_1\ A\&_\_1\ 
          \[Mu][x]\ \[Phi]\_1[x]\), "-", 
        RowBox[{"2", " ", "I", " ", \(E\^\(I\ T\_0\ \[Omega]\_2\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ A, 2]],
              Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
          \(\[Phi]\_2[x]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\_2\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ 
                  OverBar[ A], 2]],
              Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
          \(\[Phi]\_2[x]\)}], "-", 
        \(2\ I\ E\^\(I\ T\_0\ \[Omega]\_2\)\ A\_2\ \[Omega]\_2\ \[Mu][x]\ 
          \[Phi]\_2[x]\), "+", 
        \(2\ I\ E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\ \[Omega]\_2\ A\&_\_2\ 
          \[Mu][x]\ \[Phi]\_2[x]\), "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[CapitalOmega]\) + \[Omega]\_1)\)\)\), " ", 
          "F", " ", \(A\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((\[CapitalOmega] + \[Omega]\_1)\)\)\), 
          " ", "F", " ", \(A\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{\(E\^\(3\ I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((2\ \[Omega]\_1 + \[Omega]\_2)\)\)\), 
          " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
          \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((\[Omega]\_1 + 2\ \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[CapitalOmega]\) - \[Omega]\_1)\)\)\), " ", 
          "F", " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((\[CapitalOmega] - \[Omega]\_1)\)\)\), 
          " ", "F", " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "3", " ", \(E\^\(I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
          \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "4", " ", \(E\^\(I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\_2\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + 2\ \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
          \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "3", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\&_\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_1 + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
          \(A\&_\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(\(-3\)\ I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\%3\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((2\ \[Omega]\_1 - \[Omega]\_2)\)\)\), 
          " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
          \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\_2\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "4", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\&_\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
          \(A\&_\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_1 - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\%2\), " ", 
          \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((\[Omega]\_1 - 2\ \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - 2\ \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\), " ", 
          \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[CapitalOmega]\) + \[Omega]\_2)\)\)\), " ", 
          "F", " ", \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((\[CapitalOmega] + \[Omega]\_2)\)\)\), 
          " ", "F", " ", \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((2\ \[Omega]\_1 + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
          \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((\[Omega]\_1 + 2\ \[Omega]\_2)\)\)\), 
          " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{\(E\^\(3\ I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%3\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\_2\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + 2\ \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
          \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_1 + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
          \(A\&_\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[CapitalOmega]\) - \[Omega]\_2)\)\)\), " ", 
          "F", " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((\[CapitalOmega] - \[Omega]\_2)\)\)\), 
          " ", "F", " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((2\ \[Omega]\_1 - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
          \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "4", " ", \(E\^\(I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\_2\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "3", " ", \(E\^\(I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
          \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\&_\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "4", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
          \(A\&_\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_1 - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\%2\), " ", 
          \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", \(E\^\(I\ T\_0\ \((\[Omega]\_1 - 2\ \[Omega]\_2)\)\)\), 
          " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
          \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "3", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
          \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - 2\ \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\), " ", 
          \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(E\^\(\(-3\)\ I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_2\%3\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the first-order boundary conditions are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1BC = eqEps[\([2]\)] // Rest\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(w\_1[0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", \(w\_1[1, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Principal Parametric Resonance of the First Mode", "Subsubsection",
  CellFrame->False,
  CellDingbat->"\[FilledSquare]",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this case, ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalOmega] \[TildeTilde] 2  \[Omega]\_1\)]],
  ". To describe the nearness of the internal and principal parametric \
resonances, we introduce the two detuning parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Sigma]\_1\ \)\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_2\)]],
  " defined by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_2}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and define the following rules:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma]\_2 + 2\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] 
          1\/3\ \((\(-\[Epsilon]\)\ \[Sigma]\_1 + \[Omega]\_2)\), 
        \[CapitalOmega] \[Rule] 
          1\/3\ \((
              \(-2\)\ \[Epsilon]\ \[Sigma]\_1 + 3\ \[Epsilon]\ \[Sigma]\_2 + 
                2\ \[Omega]\_2)\)}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand side of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the source of secular terms as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST = 
      Table[Coefficient[order1Eq[\([2]\)] /. expRule[i], 
          Exp[I\ \[Omega]\_i\ T\_0]], {i, 2}]; \nST /. displayRule\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
            \(\[Phi]\_1[x]\)}], "-", 
          \(2\ I\ A\_1\ \[Omega]\_1\ \[Mu][x]\ \[Phi]\_1[x]\), "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
            \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"3", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
            \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
            \(A\&_\_1\%2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"2", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2\)\), 
            " ", "F", " ", \(A\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
            \(A\&_\_1\%2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"4", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}]}], ",", 
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
            \(\[Phi]\_2[x]\)}], "-", 
          \(2\ I\ A\_2\ \[Omega]\_2\ \[Mu][x]\ \[Phi]\_2[x]\), "+", 
          RowBox[{
          "2", " ", 
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2\)\), " ", 
            "F", " ", \(A\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"4", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"2", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"3", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
            \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Because the homogeneous part of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order1BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " has a nontrivial solution, the corresponding nonhomogeneous problem has a \
solution only if solvability conditions are satisfied. It follows from \
Section 7.1.1 that the homogeneous problem is self-adjoint and that the \
solvability conditions demand that ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_1[x]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_2[x]\)]],
  ", respectively. Imposing these conditions, we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[\(\(int[\[Phi]\_j[x]\ ST[\([j]\)], {x, 0, 1}] /. intRule1\) //. 
                intRule2\) /. int -> Integrate, {j, 2}] == 0 // Thread; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2\)\), 
              " ", "F", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\)}], "-", 
            RowBox[{
            "2", " ", "I", " ", 
              \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), 
              " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "-", 
            \(2\ I\ \((
                \[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_1[x]\^2\) 
                    \[DifferentialD]x)\)\ A\_1\ \[Omega]\_1\), "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\)}], 
            "+", 
            RowBox[{"3", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
              \(A\&_\_1\)}], "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
              \(A\&_\_1\%2\)}], "+", 
            RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
              \(A\&_\_1\%2\)}], "+", 
            RowBox[{"2", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_2\)}], "+", 
            RowBox[{"4", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_2\)}]}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{
            "2", " ", 
              \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2\)\), 
              " ", "F", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\)}], "+", 
            RowBox[{
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\)}], 
            "-", 
            RowBox[{
            "2", " ", "I", " ", 
              \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), 
              " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "-", 
            \(2\ I\ \((
                \[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_2[x]\^2\) 
                    \[DifferentialD]x)\)\ A\_2\ \[Omega]\_2\), "+", 
            RowBox[{"4", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_1\)}], "+", 
            RowBox[{"2", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_1\)}], "+", 
            RowBox[{"3", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
              \(A\&_\_2\)}]}], "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
To simplify the notation in the solvability conditions, we use the \
orthonormality of the mode shapes and define the following parameters:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"notationRule", "=", 
        RowBox[{"{", 
          RowBox[{
          \(\[Integral]\_0\%1\( \[Phi]\_i_[x]\^2\) \[DifferentialD]x -> 1\), 
            ",", \(\[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_i_[x]\^2\) 
                  \[DifferentialD]x -> \[Mu]\_i\), ",", 
            RowBox[{
              RowBox[{"F", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{\(\[Phi]\_i_[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "j_", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], 
                    \(\[DifferentialD]x\)}]}]}], "->", 
              \(\(-\[Omega]\_i\)\ s\_\(i, j\)\ f\)}]}], "}"}]}], ";"}]], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To identify the forms of the nonlinear terms in the solvability conditions, \
we first identify all of the possible forms of the nonhomogeneous terms in \
the first-order problem. To this end, we let\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(basicTerms = 
      Table[{A\_i[T\_1]\ E\^\(I\ \[Omega]\_i\ T\_0\), 
            A\&_\_i[T\_1]\ E\^\(\(-I\)\ \[Omega]\_i\ T\_0\)}, {i, 2}] // 
        Flatten\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(I\ T\_0\ \[Omega]\_1\)\ A\_1[T\_1], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\ A\&_\_1[T\_1], 
      E\^\(I\ T\_0\ \[Omega]\_2\)\ A\_2[T\_1], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\ A\&_\_2[T\_1]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Then, all of the possible forms of the nonlinear terms in the first-order \
problem are given by\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cubicTerms = 
      \(Nest[Outer[Times, basicTerms, #]\ &, basicTerms, 2] // Flatten\) // 
        Union; \ncubicTerms /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(3\ I\ T\_0\ \[Omega]\_1\)\ A\_1\%3, 
      E\^\(I\ T\_0\ \((2\ \[Omega]\_1 + \[Omega]\_2)\)\)\ A\_1\%2\ A\_2, 
      E\^\(I\ T\_0\ \((\[Omega]\_1 + 2\ \[Omega]\_2)\)\)\ A\_1\ A\_2\%2, 
      E\^\(3\ I\ T\_0\ \[Omega]\_2\)\ A\_2\%3, 
      E\^\(I\ T\_0\ \[Omega]\_1\)\ A\_1\%2\ A\&_\_1, 
      E\^\(I\ T\_0\ \[Omega]\_2\)\ A\_1\ A\_2\ A\&_\_1, 
      E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + 2\ \[Omega]\_2)\)\)\ A\_2\%2\ 
        A\&_\_1, E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\ A\_1\ A\&_\_1\%2, 
      E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_1 + \[Omega]\_2)\)\)\ A\_2\ 
        A\&_\_1\%2, E\^\(\(-3\)\ I\ T\_0\ \[Omega]\_1\)\ A\&_\_1\%3, 
      E\^\(I\ T\_0\ \((2\ \[Omega]\_1 - \[Omega]\_2)\)\)\ A\_1\%2\ A\&_\_2, 
      E\^\(I\ T\_0\ \[Omega]\_1\)\ A\_1\ A\_2\ A\&_\_2, 
      E\^\(I\ T\_0\ \[Omega]\_2\)\ A\_2\%2\ A\&_\_2, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\ A\_1\ A\&_\_1\ A\&_\_2, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\ A\_2\ A\&_\_1\ A\&_\_2, 
      E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_1 - \[Omega]\_2)\)\)\ A\&_\_1\%2\ 
        A\&_\_2, 
      E\^\(I\ T\_0\ \((\[Omega]\_1 - 2\ \[Omega]\_2)\)\)\ A\_1\ A\&_\_2\%2, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\ A\_2\ A\&_\_2\%2, 
      E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - 2\ \[Omega]\_2)\)\)\ A\&_\_1\ 
        A\&_\_2\%2, E\^\(\(-3\)\ I\ T\_0\ \[Omega]\_2\)\ A\&_\_2\%3}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Out of these terms, only the terms that may lead to secular terms appear in \
the solvability conditions, which can be identified according to\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(secularTerms = 
      \(\((\(\(\(E\^\(\(-I\)\ \[Omega]\_#\ T\_0\)\) cubicTerms /. 
                    expRule[#]\) /. Exp[_\ T\_0 + _. ] -> 0 // Union\) // 
              Rest)\)\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{A\_1[T\_1]\^2\ A\&_\_1[T\_1], 
        E\^\(I\ T\_1\ \[Sigma]\_1\)\ A\_2[T\_1]\ A\&_\_1[T\_1]\^2, 
        A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1]}, {
        E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ A\_1[T\_1]\^3, 
        A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1], A\_2[T\_1]\^2\ A\&_\_2[T\_1]}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Next, we define the following parameters:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolList = {
        \(-8\)\ \(\[Omega]\_1\) {\[Gamma]\_11, \[Delta]\_1, \[Gamma]\_12}, 
        \(-8\) \(\[Omega]\_2\) {\[Delta]\_2, \[Gamma]\_21, \[Gamma]\_22}}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\(-8\)\ \[Gamma]\_11\ \[Omega]\_1, \(-8\)\ \[Delta]\_1\ \[Omega]\_1, 
        \(-8\)\ \[Gamma]\_12\ \[Omega]\_1}, {
        \(-8\)\ \[Delta]\_2\ \[Omega]\_2, \(-8\)\ \[Gamma]\_21\ \[Omega]\_2, 
        \(-8\)\ \[Gamma]\_22\ \[Omega]\_2}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "and express ",
  StyleBox["SCond",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqMod = 
      \(Expand[\(-\(1\/\[Omega]\_#\)\) 
                \((\((\(SCond[\([#, 1]\)] /. notationRule\) /. 
                        Thread[secularTerms[\([#]\)] -> 0])\) + 
                    symbolList[\([#]\)] . secularTerms[\([#]\)])\)] == 0\ &
          \)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(2\ I\ \[Mu]\_1\ A\_1[T\_1]\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2\)\ f\ 
              s\_\(1, 2\)\ A\_2[T\_1]\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ f\ s\_\(1, 1\)\ 
              A\&_\_1[T\_1]\), "+", 
            \(8\ \[Gamma]\_11\ A\_1[T\_1]\^2\ A\&_\_1[T\_1]\), "+", 
            \(8\ E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Delta]\_1\ A\_2[T\_1]\ 
              A\&_\_1[T\_1]\^2\), "+", 
            \(8\ \[Gamma]\_12\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1]\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(2\ E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2\)\ f\ 
              s\_\(2, 1\)\ A\_1[T\_1]\), "+", 
            \(8\ E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ \[Delta]\_2\ 
              A\_1[T\_1]\^3\), "+", \(2\ I\ \[Mu]\_2\ A\_2[T\_1]\), "+", 
            \(8\ \[Gamma]\_21\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1]\), "+", 
            \(8\ \[Gamma]\_22\ A\_2[T\_1]\^2\ A\&_\_2[T\_1]\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(-symbolList[\([1]\)]\) -> 
          \((\(\(-Coefficient[SCond[\([1, 1]\)], #]\)\ &\)\ /@\ 
              secularTerms[\([1]\)])\) // Thread\) // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(8\ \[Gamma]\_11\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{\(-3\), " ", "\[Alpha]", " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]},
          {
            RowBox[{\(8\ \[Delta]\_1\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_12\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"4", " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {Times[ 8, 
          Subscript[ \[Gamma], 11], 
          Subscript[ \[Omega], 1]] -> Times[ -3, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 1][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], Times[ 8, 
          Subscript[ \[Delta], 1], 
          Subscript[ \[Omega], 1]] -> Plus[ 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -1, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]], Times[ 8, 
          Subscript[ \[Gamma], 12], 
          Subscript[ \[Omega], 1]] -> Plus[ 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -4, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(-symbolList[\([2]\)]\) -> 
          \((\(\(-Coefficient[SCond[\([2, 1]\)], #]\)\ &\)\ /@\ 
              secularTerms[\([2]\)])\) // Thread\) // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(8\ \[Delta]\_2\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{\(-\[Alpha]\), " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_21\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-4\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_2[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_2[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_22\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{\(-3\), " ", "\[Alpha]", " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {Times[ 8, 
          Subscript[ \[Delta], 2], 
          Subscript[ \[Omega], 2]] -> Times[ -1, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 2][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], Times[ 8, 
          Subscript[ \[Gamma], 21], 
          Subscript[ \[Omega], 2]] -> Plus[ 
          Times[ -4, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 2][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 2][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]], Times[ 8, 
          Subscript[ \[Gamma], 22], 
          Subscript[ \[Omega], 2]] -> Times[ -3, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 2]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 2][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox["Modulation Equations in Polar Form",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell["\<\
The complex-valued solvability conditions can be expressed in real-valued \
form by introducing the polar transformation\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ruleA = {A\_i_ -> \((\(1\/2\) a\_i[#]\ Exp[I\ \[Theta]\_i[#]]&)\), 
        A\&_\_i_ -> \((\(1\/2\) a\_i[#]\ Exp[\(-I\)\ \[Theta]\_i[#]]&)\)}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "into ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(expr1 = 
      \(Expand[eqMod[\([#, 1]\)]\ Exp[\(-I\)\ \[Theta]\_#[T\_1]] /. ruleA]\ &
          \)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
        \(I\ \[Mu]\_1\ a\_1[T\_1]\), "+", 
          \(E\^\(I\ T\_1\ \[Sigma]\_2 - 2\ I\ \[Theta]\_1[T\_1]\)\ f\ 
            s\_\(1, 1\)\ a\_1[T\_1]\), "+", \(\[Gamma]\_11\ a\_1[T\_1]\^3\), 
          "+", \(E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2 - 
                I\ \[Theta]\_1[T\_1] + I\ \[Theta]\_2[T\_1]\)\ f\ 
            s\_\(1, 2\)\ a\_2[T\_1]\), "+", 
          \(E\^\(I\ T\_1\ \[Sigma]\_1 - 3\ I\ \[Theta]\_1[T\_1] + 
                I\ \[Theta]\_2[T\_1]\)\ \[Delta]\_1\ a\_1[T\_1]\^2\ 
            a\_2[T\_1]\), "+", \(\[Gamma]\_12\ a\_1[T\_1]\ a\_2[T\_1]\^2\), 
          "+", 
          RowBox[{"I", " ", 
            RowBox[{
              SubsuperscriptBox["a", "1", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
          RowBox[{\(a\_1[T\_1]\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Theta]", "1", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], ",", 
        RowBox[{
        \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2 + 
                I\ \[Theta]\_1[T\_1] - I\ \[Theta]\_2[T\_1]\)\ f\ 
            s\_\(2, 1\)\ a\_1[T\_1]\), "+", 
          \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + 3\ I\ \[Theta]\_1[T\_1] - 
                I\ \[Theta]\_2[T\_1]\)\ \[Delta]\_2\ a\_1[T\_1]\^3\), "+", 
          \(I\ \[Mu]\_2\ a\_2[T\_1]\), "+", 
          \(\[Gamma]\_21\ a\_1[T\_1]\^2\ a\_2[T\_1]\), "+", 
          \(\[Gamma]\_22\ a\_2[T\_1]\^3\), "+", 
          RowBox[{"I", " ", 
            RowBox[{
              SubsuperscriptBox["a", "2", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
          RowBox[{\(a\_2[T\_1]\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Theta]", "2", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Next, we separate the real and imaginary parts of ",
  StyleBox["expr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to obtain the equations governing the modulation of the amplitudes ",
  Cell[BoxData[
      \(TraditionalForm\`a\_i\)]],
  " and phases ",
  Cell[BoxData[
      \(TraditionalForm\`\[Theta]\_i\)]],
  ". To accomplish this, we define the rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(realRule = {Re[s_] -> s, Im[s_] -> 0}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Then, the equations governing the amplitudes of motion correspond to the \
imaginary parts of ",
  StyleBox["expr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"ampEq", "=", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"Solve", "[", 
              RowBox[{
              \(\(ComplexExpand[Im[#]] == 0 /. realRule\ &\)\ /@\ expr1\), 
                ",", 
                RowBox[{"{", 
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["a", "1", "\[Prime]",
                        MultilineFunction->None], "[", \(T\_1\), "]"}], ",", 
                    RowBox[{
                      SubsuperscriptBox["a", "2", "\[Prime]",
                        MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                  "}"}]}], "]"}], "[", \([1]\), "]"}], "/.", 
          \(Rule -> Equal\)}], "//", "ExpandAll"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox["a", "1", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
          \(\(-\[Mu]\_1\)\ a\_1[T\_1] - 
            f\ Sin[T\_1\ \[Sigma]\_2 - 2\ \[Theta]\_1[T\_1]]\ s\_\(1, 1\)\ 
              a\_1[T\_1] - 
            f\ Sin[T\_1\ \[Sigma]\_1 - T\_1\ \[Sigma]\_2 - 
                  \[Theta]\_1[T\_1] + \[Theta]\_2[T\_1]]\ s\_\(1, 2\)\ 
              a\_2[T\_1] - 
            Sin[T\_1\ \[Sigma]\_1 - 3\ \[Theta]\_1[T\_1] + \[Theta]\_2[T\_1]]
                \ \[Delta]\_1\ a\_1[T\_1]\^2\ a\_2[T\_1]\)}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["a", "2", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
          \(f\ Sin[
                T\_1\ \[Sigma]\_1 - T\_1\ \[Sigma]\_2 - \[Theta]\_1[T\_1] + 
                  \[Theta]\_2[T\_1]]\ s\_\(2, 1\)\ a\_1[T\_1] + 
            Sin[T\_1\ \[Sigma]\_1 - 3\ \[Theta]\_1[T\_1] + \[Theta]\_2[T\_1]]
                \ \[Delta]\_2\ a\_1[T\_1]\^3 - \[Mu]\_2\ a\_2[T\_1]\)}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, the equations governing the phases of motion correspond to the \
real parts of ",
  StyleBox["expr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseEq = 
      \(Solve[\(ComplexExpand[Re[#]] == 0 /. realRule\ &\)\ /@\ expr1, {
                \(\[Theta]\_1\^\[Prime]\)[T\_1], 
                \(\[Theta]\_2\^\[Prime]\)[T\_1]}]\)[\([1]\)] /. 
          Rule -> Equal // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox["\[Theta]", "1", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
          \(f\ Cos[T\_1\ \[Sigma]\_2 - 2\ \[Theta]\_1[T\_1]]\ s\_\(1, 1\) + 
            \[Gamma]\_11\ a\_1[T\_1]\^2 + 
            \(f\ Cos[
                  T\_1\ \[Sigma]\_1 - T\_1\ \[Sigma]\_2 - \[Theta]\_1[T\_1] + 
                    \[Theta]\_2[T\_1]]\ s\_\(1, 2\)\ a\_2[T\_1]\)\/a\_1[
                T\_1] + 
            Cos[T\_1\ \[Sigma]\_1 - 3\ \[Theta]\_1[T\_1] + \[Theta]\_2[T\_1]]
                \ \[Delta]\_1\ a\_1[T\_1]\ a\_2[T\_1] + 
            \[Gamma]\_12\ a\_2[T\_1]\^2\)}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["\[Theta]", "2", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
          \(\[Gamma]\_21\ a\_1[T\_1]\^2 + 
            \(f\ Cos[
                  T\_1\ \[Sigma]\_1 - T\_1\ \[Sigma]\_2 - \[Theta]\_1[T\_1] + 
                    \[Theta]\_2[T\_1]]\ s\_\(2, 1\)\ a\_1[T\_1]\)\/a\_2[
                T\_1] + 
            \(Cos[T\_1\ \[Sigma]\_1 - 3\ \[Theta]\_1[T\_1] + 
                    \[Theta]\_2[T\_1]]\ \[Delta]\_2\ a\_1[T\_1]\^3\)\/a\_2[
                T\_1] + \[Gamma]\_22\ a\_2[T\_1]\^2\)}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "These modulation equations are nonautonomous because they depend \
explicitly on ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine an autonomous set of modulation equations, we start with ",
  StyleBox["expr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "and identify the independent phase arguments in it. To accomplish this, we \
first identify all possible phase arguments; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(expTerms = I\ Cases[expr1, Exp[arg_] -> arg, Infinity] // Expand\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\(-T\_1\)\ \[Sigma]\_2 + 2\ \[Theta]\_1[T\_1], 
      \(-T\_1\)\ \[Sigma]\_1 + T\_1\ \[Sigma]\_2 + \[Theta]\_1[T\_1] - 
        \[Theta]\_2[T\_1], 
      \(-T\_1\)\ \[Sigma]\_1 + 3\ \[Theta]\_1[T\_1] - \[Theta]\_2[T\_1], 
      T\_1\ \[Sigma]\_1 - T\_1\ \[Sigma]\_2 - \[Theta]\_1[T\_1] + 
        \[Theta]\_2[T\_1], 
      T\_1\ \[Sigma]\_1 - 3\ \[Theta]\_1[T\_1] + \[Theta]\_2[T\_1]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Out of these arguments, only two are independent because", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Outer[D, expTerms, {\[Theta]\_1[T\_1], \[Theta]\_2[T\_1]}] // RowReduce
      \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{1, 0}, {0, 1}, {0, 0}, {0, 0}, {0, 0}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Denoting these two independent arguments by ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_2\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(gammaList = {2  \[Gamma]\_1[T\_1], \[Gamma]\_2[T\_1]} == 
          expTerms[\([{1, 2}]\)] // Thread\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({2\ \[Gamma]\_1[T\_1] == \(-T\_1\)\ \[Sigma]\_2 + 2\ \[Theta]\_1[T\_1], 
      \[Gamma]\_2[T\_1] == 
        \(-T\_1\)\ \[Sigma]\_1 + T\_1\ \[Sigma]\_2 + \[Theta]\_1[T\_1] - 
          \[Theta]\_2[T\_1]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "which can be solved for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Theta]\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Theta]\_2\)]],
  " to obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(thetaRule = 
      \(Solve[gammaList, {\[Theta]\_1[T\_1], \[Theta]\_2[T\_1]}]\)[
          \([1]\)] // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Theta]\_1[T\_1] \[Rule] 
        \(T\_1\ \[Sigma]\_2\)\/2 + \[Gamma]\_1[T\_1], 
      \[Theta]\_2[T\_1] \[Rule] 
        \(-T\_1\)\ \[Sigma]\_1 + \(3\ T\_1\ \[Sigma]\_2\)\/2 + 
          \[Gamma]\_1[T\_1] - \[Gamma]\_2[T\_1]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting for the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Theta]\_i\)]],
  " in ",
  StyleBox["expr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(expr2 = 
      \(expr1 /. 
            Table[\[Theta]\_i -> 
                \((Evaluate[thetaRule[\([i, 2]\)] /. T\_1 -> #]\ &)\), {i, 2}]
              \) /. Exp[arg_] :> Exp[Expand[arg]] // Expand\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
        \(I\ \[Mu]\_1\ a\_1[T\_1]\), "-", \(1\/2\ \[Sigma]\_2\ a\_1[T\_1]\), 
          "+", \(E\^\(\(-2\)\ I\ \[Gamma]\_1[T\_1]\)\ f\ s\_\(1, 1\)\ 
            a\_1[T\_1]\), "+", \(\[Gamma]\_11\ a\_1[T\_1]\^3\), "+", 
          \(E\^\(\(-I\)\ \[Gamma]\_2[T\_1]\)\ f\ s\_\(1, 2\)\ a\_2[T\_1]\), 
          "+", \(E\^\(\(-2\)\ I\ \[Gamma]\_1[T\_1] - I\ \[Gamma]\_2[T\_1]\)\ 
            \[Delta]\_1\ a\_1[T\_1]\^2\ a\_2[T\_1]\), "+", 
          \(\[Gamma]\_12\ a\_1[T\_1]\ a\_2[T\_1]\^2\), "+", 
          RowBox[{"I", " ", 
            RowBox[{
              SubsuperscriptBox["a", "1", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
          RowBox[{\(a\_1[T\_1]\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Gamma]", "1", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], ",", 
        RowBox[{
        \(E\^\(I\ \[Gamma]\_2[T\_1]\)\ f\ s\_\(2, 1\)\ a\_1[T\_1]\), "+", 
          \(E\^\(2\ I\ \[Gamma]\_1[T\_1] + I\ \[Gamma]\_2[T\_1]\)\ 
            \[Delta]\_2\ a\_1[T\_1]\^3\), "+", \(I\ \[Mu]\_2\ a\_2[T\_1]\), 
          "+", \(\[Sigma]\_1\ a\_2[T\_1]\), "-", 
          \(3\/2\ \[Sigma]\_2\ a\_2[T\_1]\), "+", 
          \(\[Gamma]\_21\ a\_1[T\_1]\^2\ a\_2[T\_1]\), "+", 
          \(\[Gamma]\_22\ a\_2[T\_1]\^3\), "+", 
          RowBox[{"I", " ", 
            RowBox[{
              SubsuperscriptBox["a", "2", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
          RowBox[{\(a\_2[T\_1]\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Gamma]", "1", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
          RowBox[{\(a\_2[T\_1]\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Gamma]", "2", "\[Prime]",
                MultilineFunction->None], "[", \(T\_1\), "]"}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, the equations governing the amplitudes of motion correspond to the \
imaginary parts of ",
  StyleBox["expr2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
      RowBox[{"(", 
        RowBox[{"ampEq", "=", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{"Solve", "[", 
                  RowBox[{
                  \(\(ComplexExpand[Im[#]] == 0 /. realRule\ &\)\ /@\ expr2
                      \), ",", 
                    RowBox[{"{", 
                      RowBox[{
                        RowBox[{
                          SubsuperscriptBox["a", "1", "\[Prime]",
                            MultilineFunction->None], "[", \(T\_1\), "]"}], 
                        ",", 
                        RowBox[{
                          SubsuperscriptBox["a", "2", "\[Prime]",
                            MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                      "}"}]}], "]"}], "[", \([1]\), "]"}], "/.", 
              \(Rule -> Equal\)}], "//", "ExpandAll"}]}], ")"}], "/.", 
      \(f_[T\_1] -> f\)}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SubsuperscriptBox["a", "1", "\[Prime]",
            MultilineFunction->None], "==", 
          \(Sin[2\ \[Gamma]\_1 + \[Gamma]\_2]\ a\_1\%2\ a\_2\ \[Delta]\_1 - 
            a\_1\ \[Mu]\_1 + f\ Sin[2\ \[Gamma]\_1]\ a\_1\ s\_\(1, 1\) + 
            f\ Sin[\[Gamma]\_2]\ a\_2\ s\_\(1, 2\)\)}], ",", 
        RowBox[{
          SubsuperscriptBox["a", "2", "\[Prime]",
            MultilineFunction->None], "==", 
          \(\(-Sin[2\ \[Gamma]\_1 + \[Gamma]\_2]\)\ a\_1\%3\ \[Delta]\_2 - 
            a\_2\ \[Mu]\_2 - f\ Sin[\[Gamma]\_2]\ a\_1\ s\_\(2, 1\)\)}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, the equations governing the phases of motion correspond to the \
real parts of ",
  StyleBox["expr2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((phaseEq = 
          \(Solve[\(ComplexExpand[Re[#]] == 0 /. realRule\ &\)\ /@\ expr2, {
                    \(\[Gamma]\_1\^\[Prime]\)[T\_1], 
                    \(\[Gamma]\_2\^\[Prime]\)[T\_1]}]\)[\([1]\)] /. 
              Rule -> Equal // ExpandAll)\) /. f_[T\_1] -> f\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SubsuperscriptBox["\[Gamma]", "2", "\[Prime]",
            MultilineFunction->None], "==", 
          \(a\_1\%2\ \[Gamma]\_11 + a\_2\%2\ \[Gamma]\_12 - 
            a\_1\%2\ \[Gamma]\_21 - a\_2\%2\ \[Gamma]\_22 + 
            Cos[2\ \[Gamma]\_1 + \[Gamma]\_2]\ a\_1\ a\_2\ \[Delta]\_1 - 
            \(Cos[2\ \[Gamma]\_1 + \[Gamma]\_2]\ a\_1\%3\ \[Delta]\_2\)\/a
                \_2 - \[Sigma]\_1 + \[Sigma]\_2 + 
            f\ Cos[2\ \[Gamma]\_1]\ s\_\(1, 1\) + 
            \(f\ Cos[\[Gamma]\_2]\ a\_2\ s\_\(1, 2\)\)\/a\_1 - 
            \(f\ Cos[\[Gamma]\_2]\ a\_1\ s\_\(2, 1\)\)\/a\_2\)}], ",", 
        RowBox[{
          SubsuperscriptBox["\[Gamma]", "1", "\[Prime]",
            MultilineFunction->None], "==", 
          \(a\_1\%2\ \[Gamma]\_11 + a\_2\%2\ \[Gamma]\_12 + 
            Cos[2\ \[Gamma]\_1 + \[Gamma]\_2]\ a\_1\ a\_2\ \[Delta]\_1 - 
            \[Sigma]\_2\/2 + f\ Cos[2\ \[Gamma]\_1]\ s\_\(1, 1\) + 
            \(f\ Cos[\[Gamma]\_2]\ a\_2\ s\_\(1, 2\)\)\/a\_1\)}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The modulation equations are autonomous because they are independent of \
the independent variable ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[StyleBox[
"Modulation Equations in Cartesian Form and Symmetry Property",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell["\<\
In performing bifurcation analyses of the dynamics of the system, one might \
find it more convenient, in some cases, to represent the modulation equations \
in Cartesian rather than polar form. As a byproduct, one can also obtain the \
symmetries of the system.To this end, we introduce the Cartesian \
transformation\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(cartRule = {
        A\_k_ -> 
          \((1\/2\ \((p\_k[#] - I\ q\_k[#])\)\ Exp[I\ \[Lambda]\_k[#]]&)\), 
        A\&_\_k_ -> 
          \((1\/2\ \((p\_k[#] + I\ q\_k[#])\)\ Exp[\(-I\)\ \[Lambda]\_k[#]]&)
            \)}; \nrealRule = {Im[x_] -> 0, Re[x_] -> x, 
        \(\[Lambda]\_k_\^\[Prime]\)[T\_1] -> \[Nu]\_k}; \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_i\)]],
  " are chosen to render the resulting modulation equations autonomous. \
Substituting the ",
  StyleBox["cartRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqModCart[k_] := 
      Exp[\(-I\)\ \[Lambda]\_k[T\_1]]\ Subtract\ @@\ eqMod[\([k]\)] /. 
          cartRule // Expand; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Next, we choose the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_i\)]],
  " to render ",
  StyleBox["eqModCart",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " autonomous. To accomplish this, we identify its independent arguments by \
identifying first all of its arguments. They are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(list1 = 
      \(-I\)\ \((
            \(\(Cases[#, Exp[x_] -> x, Infinity]\ &\)\ /@\ 
                  Array[eqModCart, 2] // Flatten\) // Union)\) // Expand\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({T\_1\ \[Sigma]\_2 - 2\ \[Lambda]\_1[T\_1], 
      \(-T\_1\)\ \[Sigma]\_1 + T\_1\ \[Sigma]\_2 + \[Lambda]\_1[T\_1] - 
        \[Lambda]\_2[T\_1], 
      \(-T\_1\)\ \[Sigma]\_1 + 3\ \[Lambda]\_1[T\_1] - \[Lambda]\_2[T\_1], 
      T\_1\ \[Sigma]\_1 - T\_1\ \[Sigma]\_2 - \[Lambda]\_1[T\_1] + 
        \[Lambda]\_2[T\_1], 
      T\_1\ \[Sigma]\_1 - 3\ \[Lambda]\_1[T\_1] + \[Lambda]\_2[T\_1]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the number of independent relations in ",
  StyleBox["list1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we use the function ",
  StyleBox["RowReduce",
    FontWeight->"Bold"],
  " and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Outer[D, list1, {\[Lambda]\_1[T\_1], \[Lambda]\_2[T\_1]}] // RowReduce
      \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{1, 0}, {0, 1}, {0, 0}, {0, 0}, {0, 0}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Hence, there are only two linearly independent arguments in",
  StyleBox[" ",
    FontSlant->"Italic"],
  StyleBox["list1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". We choose the first two to determine the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_i\)]],
  ". For an invariant set of modulation equations, we set each of these \
relations to be an even multiple of \[Pi]. Solving the resulting relations \
for the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_i\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(list1 = list1[\([{1, 2}]\)]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(lambdaRule = 
      \(Solve[list1 + {m\_1, m\_2} 2  \[Pi] == 0 // Thread, 
          Table[\[Lambda]\_i[T\_1], {i, 2}]]\)[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Lambda]\_1[T\_1] \[Rule] 
        1\/2\ \((2\ \[Pi]\ m\_1 + T\_1\ \[Sigma]\_2)\), 
      \[Lambda]\_2[T\_1] \[Rule] 
        1\/2\ \((
            2\ \[Pi]\ m\_1 + 4\ \[Pi]\ m\_2 - 2\ T\_1\ \[Sigma]\_1 + 
              3\ T\_1\ \[Sigma]\_2)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`m\_i\)]],
  " are positive or negative integers. Letting ",
  Cell[BoxData[
      \(TraditionalForm\`\[Nu]\_i = \[Lambda]\_i\^\[Prime]\)]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(D[lambdaRule, T\_1] /. realRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Nu]\_1 \[Rule] \[Sigma]\_2\/2, 
      \[Nu]\_2 \[Rule] 1\/2\ \((\(-2\)\ \[Sigma]\_1 + 3\ \[Sigma]\_2)\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, separating the real and imaginary parts of ",
  StyleBox["eqModCart",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the following Cartesian form of the modulation equations:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((eqs = 
            \(Solve[Flatten[
                        Table[\(ComplexExpand[{Im[#], Re[#]}]\ &\)\ /@\ 
                              \((eqModCart[k] /. Exp[_] -> 1)\) //. 
                            realRule, {k, 2}]] == 0 // Thread, 
                  Table[{\(p\_k\^\[Prime]\)[T\_1], 
                        \(q\_k\^\[Prime]\)[T\_1]}, {k, 2}] // Flatten]\)[
                \([1]\)] /. Rule -> Equal)\) /. f_[T\_1] -> f // TableForm
      \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{
              SubsuperscriptBox["p", "1", "\[Prime]",
                MultilineFunction->None], "==", 
              \(p\_1\%2\ q\_1\ \[Gamma]\_11 + q\_1\%3\ \[Gamma]\_11 + 
                p\_2\%2\ q\_1\ \[Gamma]\_12 + q\_1\ q\_2\%2\ \[Gamma]\_12 - 
                2\ p\_1\ p\_2\ q\_1\ \[Delta]\_1 + 
                p\_1\%2\ q\_2\ \[Delta]\_1 - q\_1\%2\ q\_2\ \[Delta]\_1 - 
                p\_1\ \[Mu]\_1 - q\_1\ \[Nu]\_1 - f\ q\_1\ s\_\(1, 1\) + 
                f\ q\_2\ s\_\(1, 2\)\)}]},
          {
            RowBox[{
              SubsuperscriptBox["q", "1", "\[Prime]",
                MultilineFunction->None], "==", 
              \(\(-p\_1\%3\)\ \[Gamma]\_11 - p\_1\ q\_1\%2\ \[Gamma]\_11 - 
                p\_1\ p\_2\%2\ \[Gamma]\_12 - p\_1\ q\_2\%2\ \[Gamma]\_12 - 
                p\_1\%2\ p\_2\ \[Delta]\_1 + p\_2\ q\_1\%2\ \[Delta]\_1 - 
                2\ p\_1\ q\_1\ q\_2\ \[Delta]\_1 - q\_1\ \[Mu]\_1 + 
                p\_1\ \[Nu]\_1 - f\ p\_1\ s\_\(1, 1\) - 
                f\ p\_2\ s\_\(1, 2\)\)}]},
          {
            RowBox[{
              SubsuperscriptBox["p", "2", "\[Prime]",
                MultilineFunction->None], "==", 
              \(p\_1\%2\ q\_2\ \[Gamma]\_21 + q\_1\%2\ q\_2\ \[Gamma]\_21 + 
                p\_2\%2\ q\_2\ \[Gamma]\_22 + q\_2\%3\ \[Gamma]\_22 + 
                3\ p\_1\%2\ q\_1\ \[Delta]\_2 - q\_1\%3\ \[Delta]\_2 - 
                p\_2\ \[Mu]\_2 - q\_2\ \[Nu]\_2 + f\ q\_1\ s\_\(2, 1\)\)}]},
          {
            RowBox[{
              SubsuperscriptBox["q", "2", "\[Prime]",
                MultilineFunction->None], "==", 
              \(\(-p\_1\%2\)\ p\_2\ \[Gamma]\_21 - 
                p\_2\ q\_1\%2\ \[Gamma]\_21 - p\_2\%3\ \[Gamma]\_22 - 
                p\_2\ q\_2\%2\ \[Gamma]\_22 - p\_1\%3\ \[Delta]\_2 + 
                3\ p\_1\ q\_1\%2\ \[Delta]\_2 - q\_2\ \[Mu]\_2 + 
                p\_2\ \[Nu]\_2 - f\ p\_1\ s\_\(2, 1\)\)}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Derivative[ 1][ 
            Subscript[ p, 1]], 
          Plus[ 
            Times[ 
              Power[ 
                Subscript[ p, 1], 2], 
              Subscript[ q, 1], 
              Subscript[ \[Gamma], 11]], 
            Times[ 
              Power[ 
                Subscript[ q, 1], 3], 
              Subscript[ \[Gamma], 11]], 
            Times[ 
              Power[ 
                Subscript[ p, 2], 2], 
              Subscript[ q, 1], 
              Subscript[ \[Gamma], 12]], 
            Times[ 
              Subscript[ q, 1], 
              Power[ 
                Subscript[ q, 2], 2], 
              Subscript[ \[Gamma], 12]], 
            Times[ -2, 
              Subscript[ p, 1], 
              Subscript[ p, 2], 
              Subscript[ q, 1], 
              Subscript[ \[Delta], 1]], 
            Times[ 
              Power[ 
                Subscript[ p, 1], 2], 
              Subscript[ q, 2], 
              Subscript[ \[Delta], 1]], 
            Times[ -1, 
              Power[ 
                Subscript[ q, 1], 2], 
              Subscript[ q, 2], 
              Subscript[ \[Delta], 1]], 
            Times[ -1, 
              Subscript[ p, 1], 
              Subscript[ \[Mu], 1]], 
            Times[ -1, 
              Subscript[ q, 1], 
              Subscript[ \[Nu], 1]], 
            Times[ -1, f, 
              Subscript[ q, 1], 
              Subscript[ s, 1, 1]], 
            Times[ f, 
              Subscript[ q, 2], 
              Subscript[ s, 1, 2]]]], 
        Equal[ 
          Derivative[ 1][ 
            Subscript[ q, 1]], 
          Plus[ 
            Times[ -1, 
              Power[ 
                Subscript[ p, 1], 3], 
              Subscript[ \[Gamma], 11]], 
            Times[ -1, 
              Subscript[ p, 1], 
              Power[ 
                Subscript[ q, 1], 2], 
              Subscript[ \[Gamma], 11]], 
            Times[ -1, 
              Subscript[ p, 1], 
              Power[ 
                Subscript[ p, 2], 2], 
              Subscript[ \[Gamma], 12]], 
            Times[ -1, 
              Subscript[ p, 1], 
              Power[ 
                Subscript[ q, 2], 2], 
              Subscript[ \[Gamma], 12]], 
            Times[ -1, 
              Power[ 
                Subscript[ p, 1], 2], 
              Subscript[ p, 2], 
              Subscript[ \[Delta], 1]], 
            Times[ 
              Subscript[ p, 2], 
              Power[ 
                Subscript[ q, 1], 2], 
              Subscript[ \[Delta], 1]], 
            Times[ -2, 
              Subscript[ p, 1], 
              Subscript[ q, 1], 
              Subscript[ q, 2], 
              Subscript[ \[Delta], 1]], 
            Times[ -1, 
              Subscript[ q, 1], 
              Subscript[ \[Mu], 1]], 
            Times[ 
              Subscript[ p, 1], 
              Subscript[ \[Nu], 1]], 
            Times[ -1, f, 
              Subscript[ p, 1], 
              Subscript[ s, 1, 1]], 
            Times[ -1, f, 
              Subscript[ p, 2], 
              Subscript[ s, 1, 2]]]], 
        Equal[ 
          Derivative[ 1][ 
            Subscript[ p, 2]], 
          Plus[ 
            Times[ 
              Power[ 
                Subscript[ p, 1], 2], 
              Subscript[ q, 2], 
              Subscript[ \[Gamma], 21]], 
            Times[ 
              Power[ 
                Subscript[ q, 1], 2], 
              Subscript[ q, 2], 
              Subscript[ \[Gamma], 21]], 
            Times[ 
              Power[ 
                Subscript[ p, 2], 2], 
              Subscript[ q, 2], 
              Subscript[ \[Gamma], 22]], 
            Times[ 
              Power[ 
                Subscript[ q, 2], 3], 
              Subscript[ \[Gamma], 22]], 
            Times[ 3, 
              Power[ 
                Subscript[ p, 1], 2], 
              Subscript[ q, 1], 
              Subscript[ \[Delta], 2]], 
            Times[ -1, 
              Power[ 
                Subscript[ q, 1], 3], 
              Subscript[ \[Delta], 2]], 
            Times[ -1, 
              Subscript[ p, 2], 
              Subscript[ \[Mu], 2]], 
            Times[ -1, 
              Subscript[ q, 2], 
              Subscript[ \[Nu], 2]], 
            Times[ f, 
              Subscript[ q, 1], 
              Subscript[ s, 2, 1]]]], 
        Equal[ 
          Derivative[ 1][ 
            Subscript[ q, 2]], 
          Plus[ 
            Times[ -1, 
              Power[ 
                Subscript[ p, 1], 2], 
              Subscript[ p, 2], 
              Subscript[ \[Gamma], 21]], 
            Times[ -1, 
              Subscript[ p, 2], 
              Power[ 
                Subscript[ q, 1], 2], 
              Subscript[ \[Gamma], 21]], 
            Times[ -1, 
              Power[ 
                Subscript[ p, 2], 3], 
              Subscript[ \[Gamma], 22]], 
            Times[ -1, 
              Subscript[ p, 2], 
              Power[ 
                Subscript[ q, 2], 2], 
              Subscript[ \[Gamma], 22]], 
            Times[ -1, 
              Power[ 
                Subscript[ p, 1], 3], 
              Subscript[ \[Delta], 2]], 
            Times[ 3, 
              Subscript[ p, 1], 
              Power[ 
                Subscript[ q, 1], 2], 
              Subscript[ \[Delta], 2]], 
            Times[ -1, 
              Subscript[ q, 2], 
              Subscript[ \[Mu], 2]], 
            Times[ 
              Subscript[ p, 2], 
              Subscript[ \[Nu], 2]], 
            Times[ -1, f, 
              Subscript[ p, 1], 
              Subscript[ s, 2, 1]]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
The symmetry property of the system can also be obtained as follows:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(phase = 
      Flatten[Table[
            Mod[\(#[\([2]\)]\ &\)\ /@\ lambdaRule /. T\_1 -> 0, 2  \[Pi]], {
              m\_1, 0, 10}, {m\_2, 0, 10}], 1] // Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{0, 0}, {\[Pi], \[Pi]}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(\(rotMat[a_] = {{Cos[a], Sin[a]}, {\(-Sin[a]\), Cos[a]}}; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symmetryList = 
      \(Join\ @@\ Table[rotMat[#[\([i]\)]] . {p\_i, q\_i}, {i, 2}]\ &\)\ /@\ 
        phase\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{p\_1, q\_1, p\_2, q\_2}, {\(-p\_1\), \(-q\_1\), \(-p\_2\), \(-q\_2\)}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Hence, if ",
  Cell[BoxData[
      \(TraditionalForm\`{p\_1, q\_1, p\_2, q\_2}\)]],
  " is a solution of the modulation equations, then ",
  Cell[BoxData[
      \(TraditionalForm\`{\(-p\_1\), \(-q\_1\), \(-p\_2\), \(-q\_2\)}\)]],
  " is also another solution of these equations. If these two sets of \
solutions are the same, then the system response is symmetric; otherwise, it \
will be asymmetric."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "The Functions ",
  StyleBox["PolarForm",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["CartesianForm",
    FontColor->RGBColor[1, 0, 1]]
}], "Subsubsection",
  CellFrame->False,
  CellDingbat->"\[FilledSquare]",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Following the procedures described in the preceding section, we build two \
functions, ",
  StyleBox["PolarForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["CartesianForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", to automate the process provided that the complex dependent variables in \
the modulation equations are written in the form ",
  Cell[BoxData[
      \(TraditionalForm\`S\_i, i = 1,  ... , neq\)]],
  " and the corresponding complex conjugates are ",
  Cell[BoxData[
      \(TraditionalForm\`S\&_\_i\)]],
  ". The functions can be slightly modified to allow a more general form for \
the modulation equations. To obtain the polar form or Cartesian form of the \
complex modulation equations, we simply provide the list of modulation \
equations and ",
  Cell[BoxData[
      \(TraditionalForm\`S\)]],
  " as two required inputs in ",
  StyleBox["PolarForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " or ",
  StyleBox["CartesianForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", respectively. If ",
  Cell[BoxData[
      \(TraditionalForm\`S = A\)]],
  ", the list of modulation equations is the only required input."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(PolarForm[eqs_List, S_Symbol:  A] := \n\t
      Module[{neq = Length[eqs], t}, \n\t\t\ \ \ 
        t = \(Cases[eqs, S\_i_[a_] -> a, \[Infinity]]\)[\([1]\)]; \n\t\t\ \ \ 
        polarRule = {
            S\_k_ -> \((\(1\/2\) a\_k[#]\ Exp[I\ \[Beta]\_k[#]]\ &)\), 
            S\&_\_k_ -> \((\(1\/2\) a\_k[#]\ Exp[\(-I\)\ \[Beta]\_k[#]]&)\)}; 
        \n\ \ \ \t\trealRule = {Im[x_] -> 0, Re[x_] -> x}; \n\ \ \ \t\t
        eqModPolar[k_] := 
          Exp[\(-I\)\ \[Beta]\_k[t]]\ Subtract\ @@\ eqs[\([k]\)] /. 
              polarRule // Expand; \n\t\t\ \ \ 
        list1 = \(I\ 
                Cases[Array[eqModPolar, neq], Exp[a_] -> a, \[Infinity]] // 
              Expand\) // Union; \n\ \ \ \t\t
        newLists[list_, n_] := \n\t\t\t\ \ \ 
          With[{v = Table[Unique[], {n}]}, \n\t\t\t\t\ \ 
            ReplaceList[list, 
              Append[Flatten[\({___, Pattern[#, _]}\ &\)\ /@\ v], ___] -> 
                v]]; \n\t\ \ \ \tbetaList = Table[\[Beta]\_i[t], {i, neq}]; \n
        \ \ \ \t\tlist2 = {}; \n\t\t\ \ \ 
        Scan[If[NullSpace[Outer[Coefficient, #, betaList]] === {}, 
              list2 = \(-#\); Return[]]\ &, newLists[list1, neq]]; \n\t\t\ \ 
        If[list2 === {}, 
          Print["\<Autonomous system is not possible!!!\nThe phase list \
is:\>"]; Return[list1]\n\t\t\ \ ]; \n\ \ \ \ \ \ \ 
        betaRule1 = 
          \(Solve[list3 = Table[\[Gamma]\_i[t], {i, neq}] == list2 // Thread, 
              betaList]\)[\([1]\)]; \n\ \ \t\t
        betaRule2 = 
          Table[\[Beta]\_i -> 
              Function[t // Evaluate, 
                \[Beta]\_i[t] /. betaRule1 // Evaluate], {i, neq}]; \n\ \ \t\t
        Do[eq0[i] = 
            \(eqModPolar[i] /. betaRule2\) /. Exp[a_] :> Exp[Expand[a]]; \n
          \ \ \ \ \ \ \ \ \ 
          eq[i] = \(Solve[ComplexExpand[Im[eq0[i]]] == 0 /. realRule, 
                  \(a\_i\^\[Prime]\)[t]]\)[\([1, 1]\)] /. Rule -> Equal; \n
          \ \ \ \ \ \ \ \ \ 
          eqRe[i] = ComplexExpand[Re[eq0[i]]] == 0 /. realRule, {i, neq}\n
          \t\t\ \ ]; \n\ \ \ \ \ \ \ 
        gammapList = Table[\(\[Gamma]\_i\^\[Prime]\)[t], {i, neq}]; \n\t\t\ \ 
        eqList = 
          \(Solve[Array[eqRe, neq], gammapList]\)[\([1]\)] /. 
              Rule -> Equal // ExpandAll; \n\ \ \ \ \ \ \ 
        Do[eq[i + neq] = eqList[\([i]\)], {i, neq}]; \n
        \ \ \ \ \ \ \ {Array[eq, 2*neq], list3}\n\ ]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(CartesianForm[eqs_List, S_Symbol:  A] := \n\t
      Module[{neq = Length[eqs], t}, \n\ \ \t\t\ 
        t = \(Cases[eqs, S\_i_[a_] -> a, \[Infinity]]\)[\([1]\)]; \n\t\t\ \ \ 
        cartRule = {
            S\_k_ -> 
              \((1/2 \((p\_k[#] - I\ q\_k[#])\)\ Exp[I\ \[Lambda]\_k[#]]\ &)
                \), S\&_\_k_ -> 
              \((1/2 \((p\_k[#] + I\ q\_k[#])\)\ 
                    Exp[\(-I\)\ \[Lambda]\_k[#]]\ &)\)}; \n\ \ \ \t\t
        realRule = 
          \ {Im[x_] -> 0, Re[x_] -> x, 
            \(\[Lambda]\_k_\^\[Prime]\)[t] -> \[Nu]\_k}; \n\ \t\ \ \t
        eqModCart[k_] := 
          Exp[\(-I\)\ \[Lambda]\_k[t]]\ Subtract\ @@\ eqs[\([k]\)] /. 
              cartRule // Expand; \n\ \ \ \t\t
        eqs1 = \({ComplexExpand[Im[#]] == 0, ComplexExpand[Re[#]] == 0}\ &
                  \)\ /@\ \((Array[eqModCart, neq] /. E^_ -> 1)\) //. 
              realRule // Flatten; \n\ \ \ \t\ 
        eqs2 = \(Solve[eqs1, 
                Table[{\(p\_k\^\[Prime]\)[t], \(q\_k\^\[Prime]\)[t]}, {k, 
                      neq}] // Flatten]\)[\([1]\)] /. Rule -> Equal; \t\t\n
        \t\t\ \ list1 = 
          \(I\ Cases[Array[eqModCart, neq], Exp[a_] -> a, \[Infinity]] // 
              Expand\) // Union; \n\t\t\ \ 
        newLists[list_, n_] := \n\t\t\t\ \ \ 
          With[{v = Table[Unique[], {n}]}, \n\t\t\t\t\ \ 
            ReplaceList[list, 
              Append[Flatten[\({___, Pattern[#, _]}\ &\)\ /@\ v], ___] -> 
                v]]; \n\t\ \ \ \ 
        lambdaList = Table[\[Lambda]\_i[t], {i, neq}]; \n\ \ \ \t\ 
        list2 = {}; \n\t\t\ \ 
        Scan[If[NullSpace[Outer[Coefficient, #, lambdaList]] === {}, 
              list2 = \(-#\); Return[]]\ &, newLists[list1, neq]]; \n\t\t\ \ 
        If[list2 === {}, 
          Print["\<Autonomous system is not possible!!!\nThe phase list \
is:\>"]; Return[list1]\n\t\t\ \ ]; \n\ \ \t\t
        lambdaRule = 
          \(Solve[list2 + 2\ Table[m\_i, {i, neq}]\ \[Pi] == 0 // Thread, 
              lambdaList]\)[\([1]\)]; \n\t\t\ \ 
        iterList = Table[{m\_i, 0, 10}, {i, neq}]; \n\t\ \ \t
        phase = Flatten[
              Table[Mod[
                    \(Select[#, \(! FreeQ[#, \[Pi]]\)\ &]\ &\)\ /@\ 
                      \((lambdaList /. lambdaRule // Expand)\), 2  \[Pi]] // 
                  Evaluate, Sequence\ @@\ iterList // Evaluate], neq - 1] // 
            Union; \n\ \ \t\t
        rotMat[\[Theta]_] := {{Cos[\[Theta]], \(-Sin[\[Theta]]\)}, {
              Sin[\[Theta]], Cos[\[Theta]]}}; \n\t\t\ \ 
        symmetryList = 
          \(Flatten[Table[rotMat[#[\([k]\)]] . {p\_k, q\_k}, {k, 2}]]\ &\)\ /@
            \ phase; \n
        \ \t\ \ \ {eqs2, D[lambdaRule, t] /. realRule, symmetryList}\n\ ]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Principal Parametric Resonance of the Second Mode", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this case, ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalOmega] \[TildeTilde] \[Omega]\_2\)]],
  ". To describe the nearness of the internal and principal parametric \
resonances, we introduce the two detuning parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Sigma]\_1\ \)\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_2\)]],
  " defined by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\ \[Sigma]\_2}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and define the following rules:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
        \[CapitalOmega] \[Rule] 
          2\ \[Epsilon]\ \[Sigma]\_1 + \[Epsilon]\ \[Sigma]\_2 + 
            6\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] 
          1\/3\ \((\(-\[Epsilon]\)\ \[Sigma]\_1 + \[Omega]\_2)\), 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma]\_2 + 2\ \[Omega]\_2}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand side of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the source of secular terms as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST = 
      Table[Coefficient[order1Eq[\([2]\)] /. expRule[i], 
          Exp[I\ \[Omega]\_i\ T\_0]], {i, 2}]; \nST /. displayRule\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
            \(\[Phi]\_1[x]\)}], "-", 
          \(2\ I\ A\_1\ \[Omega]\_1\ \[Mu][x]\ \[Phi]\_1[x]\), "+", 
          RowBox[{"3", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
            \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
            \(A\&_\_1\%2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"2", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
            \(A\&_\_1\%2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"4", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}]}], ",", 
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
            \(\[Phi]\_2[x]\)}], "-", 
          \(2\ I\ A\_2\ \[Omega]\_2\ \[Mu][x]\ \[Phi]\_2[x]\), "+", 
          RowBox[{
          \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"4", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"2", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
            \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"3", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
            \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "As discussed in the case of principal parametric resonance of the first \
mode, the solvability conditions of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order1BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demand that ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_1[x]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_2[x]\)]],
  ", respectively. Imposing these conditions, we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[\(\(int[\[Phi]\_j[x]\ ST[\([j]\)], {x, 0, 1}] /. intRule1\) //. 
                intRule2\) /. int -> Integrate, {j, 2}] == 0 // Thread; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{
            \(-2\), " ", "I", " ", 
              \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), 
              " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "-", 
            \(2\ I\ \((
                \[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_1[x]\^2\) 
                    \[DifferentialD]x)\)\ A\_1\ \[Omega]\_1\), "+", 
            RowBox[{"3", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
              \(A\&_\_1\)}], "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
              \(A\&_\_1\%2\)}], "+", 
            RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
              \(A\&_\_1\%2\)}], "+", 
            RowBox[{"2", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_2\)}], "+", 
            RowBox[{"4", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_2\)}]}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\)}], 
            "-", 
            RowBox[{
            "2", " ", "I", " ", 
              \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), 
              " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "-", 
            \(2\ I\ \((
                \[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_2[x]\^2\) 
                    \[DifferentialD]x)\)\ A\_2\ \[Omega]\_2\), "+", 
            RowBox[{"4", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_1\)}], "+", 
            RowBox[{"2", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_1\)}], "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_2\)}], 
            "+", 
            RowBox[{"3", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
              \(A\&_\_2\)}]}], "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using the notations introduced in the preceding section, we can express ",
  StyleBox["SCond",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqMod = 
      \(Expand[\(-\(1\/\[Omega]\_#\)\) 
                \((\((\(SCond[\([#, 1]\)] /. notationRule\) /. 
                        Thread[secularTerms[\([#]\)] -> 0])\) + 
                    symbolList[\([#]\)] . secularTerms[\([#]\)])\)] == 0\ &
          \)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(2\ I\ \[Mu]\_1\ A\_1[T\_1]\), "+", 
            \(8\ \[Gamma]\_11\ A\_1[T\_1]\^2\ A\&_\_1[T\_1]\), "+", 
            \(8\ E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Delta]\_1\ A\_2[T\_1]\ 
              A\&_\_1[T\_1]\^2\), "+", 
            \(8\ \[Gamma]\_12\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1]\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(8\ E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ \[Delta]\_2\ 
              A\_1[T\_1]\^3\), "+", \(2\ I\ \[Mu]\_2\ A\_2[T\_1]\), "+", 
            \(8\ \[Gamma]\_21\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1]\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ f\ s\_\(2, 2\)\ 
              A\&_\_2[T\_1]\), "+", 
            \(8\ \[Gamma]\_22\ A\_2[T\_1]\^2\ A\&_\_2[T\_1]\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(-symbolList[\([1]\)]\) -> 
          \((\(\(-Coefficient[SCond[\([1, 1]\)], #]\)\ &\)\ /@\ 
              secularTerms[\([1]\)])\) // Thread\) // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(8\ \[Gamma]\_11\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{\(-3\), " ", "\[Alpha]", " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]},
          {
            RowBox[{\(8\ \[Delta]\_1\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_12\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"4", " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {Times[ 8, 
          Subscript[ \[Gamma], 11], 
          Subscript[ \[Omega], 1]] -> Times[ -3, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 1][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], Times[ 8, 
          Subscript[ \[Delta], 1], 
          Subscript[ \[Omega], 1]] -> Plus[ 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -1, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]], Times[ 8, 
          Subscript[ \[Gamma], 12], 
          Subscript[ \[Omega], 1]] -> Plus[ 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -4, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(-symbolList[\([2]\)]\) -> 
          \((\(\(-Coefficient[SCond[\([2, 1]\)], #]\)\ &\)\ /@\ 
              secularTerms[\([2]\)])\) // Thread\) // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(8\ \[Delta]\_2\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{\(-\[Alpha]\), " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_21\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-4\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_2[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_2[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_22\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{\(-3\), " ", "\[Alpha]", " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {Times[ 8, 
          Subscript[ \[Delta], 2], 
          Subscript[ \[Omega], 2]] -> Times[ -1, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 2][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], Times[ 8, 
          Subscript[ \[Gamma], 21], 
          Subscript[ \[Omega], 2]] -> Plus[ 
          Times[ -4, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 2][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 2][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]], Times[ 8, 
          Subscript[ \[Gamma], 22], 
          Subscript[ \[Omega], 2]] -> Times[ -3, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 2]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 2][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox["Modulation Equations in Polar Form",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell[TextData[{
  "Using ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and the function ",
  StyleBox["PolarForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " defined in the preceding section, we obtain the modulation equations in \
polar form and the definitions for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_i\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(PolarForm[eqMod]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["a", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_1\)\ a\_1[T\_1] - 
                Sin[\[Gamma]\_1[T\_1]]\ \[Delta]\_1\ a\_1[T\_1]\^2\ 
                  a\_2[T\_1]\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["a", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(Sin[\[Gamma]\_1[T\_1]]\ \[Delta]\_2\ a\_1[T\_1]\^3 - 
                \[Mu]\_2\ a\_2[T\_1] - 
                f\ Sin[\[Gamma]\_2[T\_1]]\ s\_\(2, 2\)\ a\_2[T\_1]\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Gamma]", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Sigma]\_1 + f\ Cos[\[Gamma]\_2[T\_1]]\ s\_\(2, 2\) - 
                3\ \[Gamma]\_11\ a\_1[T\_1]\^2 + 
                \[Gamma]\_21\ a\_1[T\_1]\^2 + 
                \(Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_2\ a\_1[T\_1]\^3\)\/a\_2[
                    T\_1] - 
                3\ Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_1\ a\_1[T\_1]\ 
                  a\_2[T\_1] - 3\ \[Gamma]\_12\ a\_2[T\_1]\^2 + 
                \[Gamma]\_22\ a\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Gamma]", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Sigma]\_2 - 2\ f\ Cos[\[Gamma]\_2[T\_1]]\ s\_\(2, 2\) - 
                2\ \[Gamma]\_21\ a\_1[T\_1]\^2 - 
                \(2\ Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_2\ 
                    a\_1[T\_1]\^3\)\/a\_2[T\_1] - 
                2\ \[Gamma]\_22\ a\_2[T\_1]\^2\)}]}], "}"}], ",", 
        \({\[Gamma]\_1[T\_1] == 
            T\_1\ \[Sigma]\_1 - 3\ \[Beta]\_1[T\_1] + \[Beta]\_2[T\_1], 
          \[Gamma]\_2[T\_1] == T\_1\ \[Sigma]\_2 - 2\ \[Beta]\_2[T\_1]}\)}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox[
"Modulation Equations in Cartesian Form and Symmetry Property",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell[TextData[{
  "Using ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and the function ",
  StyleBox["CartesianForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " defined in the preceding section, we obtain the modulation equations in \
Cartesian form, the definitions for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Nu]\_i\)]],
  ", and the symmetry property as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(CartesianForm[eqMod]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["p", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_1\)\ p\_1[T\_1] - \[Nu]\_1\ q\_1[T\_1] + 
                \[Gamma]\_11\ p\_1[T\_1]\^2\ q\_1[T\_1] - 
                2\ \[Delta]\_1\ p\_1[T\_1]\ p\_2[T\_1]\ q\_1[T\_1] + 
                \[Gamma]\_12\ p\_2[T\_1]\^2\ q\_1[T\_1] + 
                \[Gamma]\_11\ q\_1[T\_1]\^3 + 
                \[Delta]\_1\ p\_1[T\_1]\^2\ q\_2[T\_1] - 
                \[Delta]\_1\ q\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_12\ q\_1[T\_1]\ q\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["q", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Nu]\_1\ p\_1[T\_1] - \[Gamma]\_11\ p\_1[T\_1]\^3 - 
                \[Delta]\_1\ p\_1[T\_1]\^2\ p\_2[T\_1] - 
                \[Gamma]\_12\ p\_1[T\_1]\ p\_2[T\_1]\^2 - 
                \[Mu]\_1\ q\_1[T\_1] - 
                \[Gamma]\_11\ p\_1[T\_1]\ q\_1[T\_1]\^2 + 
                \[Delta]\_1\ p\_2[T\_1]\ q\_1[T\_1]\^2 - 
                2\ \[Delta]\_1\ p\_1[T\_1]\ q\_1[T\_1]\ q\_2[T\_1] - 
                \[Gamma]\_12\ p\_1[T\_1]\ q\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["p", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_2\)\ p\_2[T\_1] + 
                3\ \[Delta]\_2\ p\_1[T\_1]\^2\ q\_1[T\_1] - 
                \[Delta]\_2\ q\_1[T\_1]\^3 - \[Nu]\_2\ q\_2[T\_1] - 
                f\ s\_\(2, 2\)\ q\_2[T\_1] + 
                \[Gamma]\_21\ p\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_22\ p\_2[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_21\ q\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_22\ q\_2[T\_1]\^3\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["q", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Delta]\_2\)\ p\_1[T\_1]\^3 + \[Nu]\_2\ p\_2[T\_1] - 
                f\ s\_\(2, 2\)\ p\_2[T\_1] - 
                \[Gamma]\_21\ p\_1[T\_1]\^2\ p\_2[T\_1] - 
                \[Gamma]\_22\ p\_2[T\_1]\^3 + 
                3\ \[Delta]\_2\ p\_1[T\_1]\ q\_1[T\_1]\^2 - 
                \[Gamma]\_21\ p\_2[T\_1]\ q\_1[T\_1]\^2 - 
                \[Mu]\_2\ q\_2[T\_1] - 
                \[Gamma]\_22\ p\_2[T\_1]\ q\_2[T\_1]\^2\)}]}], "}"}], ",", 
        \({\[Nu]\_1 \[Rule] 1\/6\ \((2\ \[Sigma]\_1 + \[Sigma]\_2)\), 
          \[Nu]\_2 \[Rule] \[Sigma]\_2\/2}\), ",", 
        \({{p\_1, q\_1, p\_2, q\_2}, {p\_1\/2 - \(\@3\ q\_1\)\/2, 
            \(\@3\ p\_1\)\/2 + q\_1\/2, \(-p\_2\), \(-q\_2\)}, {
            \(-\(p\_1\/2\)\) - \(\@3\ q\_1\)\/2, \(\@3\ p\_1\)\/2 - q\_1\/2, 
            p\_2, q\_2}, {\(-p\_1\), \(-q\_1\), \(-p\_2\), \(-q\_2\)}, {
            \(-\(p\_1\/2\)\) + \(\@3\ q\_1\)\/2, 
            \(-\(1\/2\)\)\ \@3\ p\_1 - q\_1\/2, p\_2, q\_2}, {
            p\_1\/2 + \(\@3\ q\_1\)\/2, \(-\(1\/2\)\)\ \@3\ p\_1 + q\_1\/2, 
            \(-p\_2\), \(-q\_2\)}}\)}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Combination Parametric Resonance of the Two Modes", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this case, ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[CapitalOmega] \[TildeTilde] \[Omega]\_1 + \[Omega]\_2\)]],
  ". To describe the nearness of the internal and combination parametric \
resonances, we introduce the two detuning parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Sigma]\_1\ \)\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_2\)]],
  " defined by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
        \[CapitalOmega] == 
          \[Omega]\_1 + \[Omega]\_2 + \[Epsilon]\ \[Sigma]\_2}; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell["and define the following rules:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
        \[CapitalOmega] \[Rule] 
          \[Epsilon]\ \[Sigma]\_1 + \[Epsilon]\ \[Sigma]\_2 + 
            4\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] 
          1\/3\ \((\(-\[Epsilon]\)\ \[Sigma]\_1 + \[Omega]\_2)\), 
        \[CapitalOmega] \[Rule] 
          1\/3\ \((
              \(-\[Epsilon]\)\ \[Sigma]\_1 + 3\ \[Epsilon]\ \[Sigma]\_2 + 
                4\ \[Omega]\_2)\)}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand side of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the source of secular terms as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST = 
      Table[Coefficient[order1Eq[\([2]\)] /. expRule[i], 
          Exp[I\ \[Omega]\_i\ T\_0]], {i, 2}]; \nST /. displayRule\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
            \(\[Phi]\_1[x]\)}], "-", 
          \(2\ I\ A\_1\ \[Omega]\_1\ \[Mu][x]\ \[Phi]\_1[x]\), "+", 
          RowBox[{"3", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
            \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
            \(A\&_\_1\%2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"2", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
            \(A\&_\_1\%2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
            \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"4", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}]}], ",", 
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
            \(\[Phi]\_2[x]\)}], "-", 
          \(2\ I\ A\_2\ \[Omega]\_2\ \[Mu][x]\ \[Phi]\_2[x]\), "+", 
          RowBox[{
          \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{
          "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
            \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"4", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"2", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
            \(A\_2\), " ", \(A\&_\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}], "+", 
          RowBox[{"3", " ", "\[Alpha]", " ", 
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
            \(A\&_\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "x", "]"}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The solvability conditions demand that ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to solutions of the adjoint homogeneous problem. Since the \
problem is self-adjoint, we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[\(\(int[\[Phi]\_j[x]\ ST[\([j]\)], {x, 0, 1}] /. intRule1\) //. 
                intRule2\) /. int -> Integrate, {j, 2}] == 0 // Thread; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{
            \(-2\), " ", "I", " ", 
              \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), 
              " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "-", 
            \(2\ I\ \((
                \[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_1[x]\^2\) 
                    \[DifferentialD]x)\)\ A\_1\ \[Omega]\_1\), "+", 
            RowBox[{"3", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%2\), " ", 
              \(A\&_\_1\)}], "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
              \(A\&_\_1\%2\)}], "+", 
            RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\), " ", 
              \(A\&_\_1\%2\)}], "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_2\)}], 
            "+", 
            RowBox[{"2", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_2\)}], "+", 
            RowBox[{"4", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_2\)}]}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\%3\)}], 
            "-", 
            RowBox[{
            "2", " ", "I", " ", 
              \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), 
              " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "-", 
            \(2\ I\ \((
                \[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_2[x]\^2\) 
                    \[DifferentialD]x)\)\ A\_2\ \[Omega]\_2\), "+", 
            RowBox[{
            "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1\)}], 
            "+", 
            RowBox[{"4", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_1\)}], "+", 
            RowBox[{"2", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1\), " ", 
              \(A\_2\), " ", \(A\&_\_1\)}], "+", 
            RowBox[{"3", " ", "\[Alpha]", " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", 
              RowBox[{"(", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2\%2\), " ", 
              \(A\&_\_2\)}]}], "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using the notations introduced in the preceding section, we can express ",
  StyleBox["SCond",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqMod = 
      \(Expand[\(-\(1\/\[Omega]\_#\)\) 
                \((\((\(SCond[\([#, 1]\)] /. notationRule\) /. 
                        Thread[secularTerms[\([#]\)] -> 0])\) + 
                    symbolList[\([#]\)] . secularTerms[\([#]\)])\)] == 0\ &
          \)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(2\ I\ \[Mu]\_1\ A\_1[T\_1]\), "+", 
            \(8\ \[Gamma]\_11\ A\_1[T\_1]\^2\ A\&_\_1[T\_1]\), "+", 
            \(8\ E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Delta]\_1\ A\_2[T\_1]\ 
              A\&_\_1[T\_1]\^2\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ f\ s\_\(1, 2\)\ 
              A\&_\_2[T\_1]\), "+", 
            \(8\ \[Gamma]\_12\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1]\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(8\ E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ \[Delta]\_2\ 
              A\_1[T\_1]\^3\), "+", \(2\ I\ \[Mu]\_2\ A\_2[T\_1]\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ f\ s\_\(2, 1\)\ 
              A\&_\_1[T\_1]\), "+", 
            \(8\ \[Gamma]\_21\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1]\), "+", 
            \(8\ \[Gamma]\_22\ A\_2[T\_1]\^2\ A\&_\_2[T\_1]\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(-symbolList[\([1]\)]\) -> 
          \((\(\(-Coefficient[SCond[\([1, 1]\)], #]\)\ &\)\ /@\ 
              secularTerms[\([1]\)])\) // Thread\) // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(8\ \[Gamma]\_11\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{\(-3\), " ", "\[Alpha]", " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_1[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]},
          {
            RowBox[{\(8\ \[Delta]\_1\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_12\ \[Omega]\_1\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"4", " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_1[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {Times[ 8, 
          Subscript[ \[Gamma], 11], 
          Subscript[ \[Omega], 1]] -> Times[ -3, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 1][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], Times[ 8, 
          Subscript[ \[Delta], 1], 
          Subscript[ \[Omega], 1]] -> Plus[ 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -1, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]], Times[ 8, 
          Subscript[ \[Gamma], 12], 
          Subscript[ \[Omega], 1]] -> Plus[ 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -4, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 1][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(-symbolList[\([2]\)]\) -> 
          \((\(\(-Coefficient[SCond[\([2, 1]\)], #]\)\ &\)\ /@\ 
              secularTerms[\([2]\)])\) // Thread\) // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(8\ \[Delta]\_2\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{\(-\[Alpha]\), " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_21\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-4\), " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_2[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "1", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", "\[Alpha]", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_2[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}]}]}]},
          {
            RowBox[{\(8\ \[Gamma]\_22\ \[Omega]\_2\), "\[Rule]", 
              RowBox[{\(-3\), " ", "\[Alpha]", " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_2[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "2", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {Times[ 8, 
          Subscript[ \[Delta], 2], 
          Subscript[ \[Omega], 2]] -> Times[ -1, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 2][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], Times[ 8, 
          Subscript[ \[Gamma], 21], 
          Subscript[ \[Omega], 2]] -> Plus[ 
          Times[ -4, \[Alpha], 
            Integrate[ 
              Times[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 2][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 1]][ x]], {x, 0, 1}]], 
          Times[ -2, \[Alpha], 
            Integrate[ 
              Power[ 
                Derivative[ 1][ 
                  Subscript[ \[Phi], 1]][ x], 2], {x, 0, 1}], 
            Integrate[ 
              Times[ 
                Subscript[ \[Phi], 2][ x], 
                Derivative[ 2][ 
                  Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]], Times[ 8, 
          Subscript[ \[Gamma], 22], 
          Subscript[ \[Omega], 2]] -> Times[ -3, \[Alpha], 
          Integrate[ 
            Power[ 
              Derivative[ 1][ 
                Subscript[ \[Phi], 2]][ x], 2], {x, 0, 1}], 
          Integrate[ 
            Times[ 
              Subscript[ \[Phi], 2][ x], 
              Derivative[ 2][ 
                Subscript[ \[Phi], 2]][ x]], {x, 0, 1}]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox["Modulation Equations in Polar Form",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell[TextData[{
  "Using ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and the function ",
  StyleBox["PolarForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the modulation equations in polar form and the definitions for \
",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_i\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(PolarForm[eqMod]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["a", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_1\)\ a\_1[T\_1] - 
                f\ Sin[\[Gamma]\_2[T\_1]]\ s\_\(1, 2\)\ a\_2[T\_1] - 
                Sin[\[Gamma]\_1[T\_1]]\ \[Delta]\_1\ a\_1[T\_1]\^2\ 
                  a\_2[T\_1]\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["a", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-f\)\ Sin[\[Gamma]\_2[T\_1]]\ s\_\(2, 1\)\ a\_1[T\_1] + 
                Sin[\[Gamma]\_1[T\_1]]\ \[Delta]\_2\ a\_1[T\_1]\^3 - 
                \[Mu]\_2\ a\_2[T\_1]\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Gamma]", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Sigma]\_1 - 3\ \[Gamma]\_11\ a\_1[T\_1]\^2 + 
                \[Gamma]\_21\ a\_1[T\_1]\^2 + 
                \(f\ Cos[\[Gamma]\_2[T\_1]]\ s\_\(2, 1\)\ a\_1[T\_1]\)\/a\_2[
                    T\_1] + 
                \(Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_2\ a\_1[T\_1]\^3\)\/a\_2[
                    T\_1] - 
                \(3\ f\ Cos[\[Gamma]\_2[T\_1]]\ s\_\(1, 2\)\ 
                    a\_2[T\_1]\)\/a\_1[T\_1] - 
                3\ Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_1\ a\_1[T\_1]\ 
                  a\_2[T\_1] - 3\ \[Gamma]\_12\ a\_2[T\_1]\^2 + 
                \[Gamma]\_22\ a\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Gamma]", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Sigma]\_2 - \[Gamma]\_11\ a\_1[T\_1]\^2 - 
                \[Gamma]\_21\ a\_1[T\_1]\^2 - 
                \(f\ Cos[\[Gamma]\_2[T\_1]]\ s\_\(2, 1\)\ a\_1[T\_1]\)\/a\_2[
                    T\_1] - 
                \(Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_2\ a\_1[T\_1]\^3\)\/a\_2[
                    T\_1] - 
                \(f\ Cos[\[Gamma]\_2[T\_1]]\ s\_\(1, 2\)\ a\_2[T\_1]\)\/a\_1[
                    T\_1] - 
                Cos[\[Gamma]\_1[T\_1]]\ \[Delta]\_1\ a\_1[T\_1]\ a\_2[T\_1] - 
                \[Gamma]\_12\ a\_2[T\_1]\^2 - 
                \[Gamma]\_22\ a\_2[T\_1]\^2\)}]}], "}"}], ",", 
        \({\[Gamma]\_1[T\_1] == 
            T\_1\ \[Sigma]\_1 - 3\ \[Beta]\_1[T\_1] + \[Beta]\_2[T\_1], 
          \[Gamma]\_2[T\_1] == 
            T\_1\ \[Sigma]\_2 - \[Beta]\_1[T\_1] - \[Beta]\_2[T\_1]}\)}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox[
"Modulation Equations in Cartesian Form and Symmetry Property",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell[TextData[{
  "Using ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and the function ",
  StyleBox["CartesianForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the modulation equations in Cartesian form, the definitions \
for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Nu]\_i\)]],
  ", and the symmetry property as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(CartesianForm[eqMod]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["p", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_1\)\ p\_1[T\_1] - \[Nu]\_1\ q\_1[T\_1] + 
                \[Gamma]\_11\ p\_1[T\_1]\^2\ q\_1[T\_1] - 
                2\ \[Delta]\_1\ p\_1[T\_1]\ p\_2[T\_1]\ q\_1[T\_1] + 
                \[Gamma]\_12\ p\_2[T\_1]\^2\ q\_1[T\_1] + 
                \[Gamma]\_11\ q\_1[T\_1]\^3 - f\ s\_\(1, 2\)\ q\_2[T\_1] + 
                \[Delta]\_1\ p\_1[T\_1]\^2\ q\_2[T\_1] - 
                \[Delta]\_1\ q\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_12\ q\_1[T\_1]\ q\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["q", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Nu]\_1\ p\_1[T\_1] - \[Gamma]\_11\ p\_1[T\_1]\^3 - 
                f\ s\_\(1, 2\)\ p\_2[T\_1] - 
                \[Delta]\_1\ p\_1[T\_1]\^2\ p\_2[T\_1] - 
                \[Gamma]\_12\ p\_1[T\_1]\ p\_2[T\_1]\^2 - 
                \[Mu]\_1\ q\_1[T\_1] - 
                \[Gamma]\_11\ p\_1[T\_1]\ q\_1[T\_1]\^2 + 
                \[Delta]\_1\ p\_2[T\_1]\ q\_1[T\_1]\^2 - 
                2\ \[Delta]\_1\ p\_1[T\_1]\ q\_1[T\_1]\ q\_2[T\_1] - 
                \[Gamma]\_12\ p\_1[T\_1]\ q\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["p", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_2\)\ p\_2[T\_1] - f\ s\_\(2, 1\)\ q\_1[T\_1] + 
                3\ \[Delta]\_2\ p\_1[T\_1]\^2\ q\_1[T\_1] - 
                \[Delta]\_2\ q\_1[T\_1]\^3 - \[Nu]\_2\ q\_2[T\_1] + 
                \[Gamma]\_21\ p\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_22\ p\_2[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_21\ q\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Gamma]\_22\ q\_2[T\_1]\^3\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["q", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-f\)\ s\_\(2, 1\)\ p\_1[T\_1] - 
                \[Delta]\_2\ p\_1[T\_1]\^3 + \[Nu]\_2\ p\_2[T\_1] - 
                \[Gamma]\_21\ p\_1[T\_1]\^2\ p\_2[T\_1] - 
                \[Gamma]\_22\ p\_2[T\_1]\^3 + 
                3\ \[Delta]\_2\ p\_1[T\_1]\ q\_1[T\_1]\^2 - 
                \[Gamma]\_21\ p\_2[T\_1]\ q\_1[T\_1]\^2 - 
                \[Mu]\_2\ q\_2[T\_1] - 
                \[Gamma]\_22\ p\_2[T\_1]\ q\_2[T\_1]\^2\)}]}], "}"}], ",", 
        \({\[Nu]\_1 \[Rule] 1\/4\ \((\[Sigma]\_1 + \[Sigma]\_2)\), 
          \[Nu]\_2 \[Rule] 1\/4\ \((\(-\[Sigma]\_1\) + 3\ \[Sigma]\_2)\)}\), 
        ",", \({{p\_1, q\_1, p\_2, q\_2}, {\(-q\_1\), p\_1, q\_2, 
            \(-p\_2\)}, {\(-p\_1\), \(-q\_1\), \(-p\_2\), \(-q\_2\)}, {q\_1, 
            \(-p\_1\), \(-q\_2\), p\_2}}\)}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.2.3 Discretization of the Continuous Problem", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "As an alternative, we apply the method of multiple scales to the \
discretized system of ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To determine the discretized form, we expand ",
  Cell[BoxData[
      \(TraditionalForm\`w[x, t]\)]],
  " in terms of the linear mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_m[x]\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(wRule1 = 
      w -> Function[{x, t}, 
          \[Sum]\+\(m = 1\)\%\[Infinity] u\_m[t]\ \[Phi]\_m[x]]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To simplify the computation, we temporarily drop the \[Sum] sign in ",
  StyleBox["wRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", as long as we know the repeated ",
  Cell[BoxData[
      \(TraditionalForm\`m\)]],
  " represents a summation index, and rewrite it as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(wRule2 = w -> Function[{x, t}, u\_m[t]\ \[Phi]\_m[x]]; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`u\_m[t]\)]],
  " are the generalized coordinates. Substituting ",
  StyleBox["wRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", multiplying the result with ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_n[x]\)]],
  ", and integrating the outcome from ",
  Cell[BoxData[
      \(TraditionalForm\`x = 0\)]],
  " to ",
  Cell[BoxData[
      \(TraditionalForm\`x = 1\)]],
  " using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the following discretized form of the equations describing the \
response of the beam:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq723a = 
      \(\(\(int[\[Phi]\_n[x]\ \((\(# /. Integrate -> int\) /. wRule2)\), {x, 
                    0, 1}] /. intRule1\) //. intRule2\) /. int -> Integrate\ &
          \)\ /@\ EOM\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(-2\), " ", "P", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{\(\[Phi]\_n[x]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(u\_m[t]\)}], "+", 
        RowBox[{
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{\(\[Phi]\_n[x]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "m", 
                      TagBox[\((4)\),
                        Derivative],
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(u\_m[t]\)}], "+", 
        RowBox[{
        \((\[Integral]\_0\%1\( \[Phi]\_m[x]\ \[Phi]\_n[x]\) \[DifferentialD]x)
            \), " ", 
          RowBox[{
            SubsuperscriptBox["u", "m", "\[DoublePrime]",
              MultilineFunction->None], "[", "t", "]"}]}]}], "==", 
      RowBox[{
        RowBox[{
        "4", " ", "F", " ", "\[Epsilon]", " ", \(Cos[t\ \[CapitalOmega]]\), 
          " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{\(\[Phi]\_n[x]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(u\_m[t]\)}], "+", 
        RowBox[{"\[Alpha]", " ", "\[Epsilon]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{\(\[Phi]\_n[x]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(u\_m[t]\^3\)}], 
        "-", 
        RowBox[{
        "2", " ", "\[Epsilon]", " ", 
          \((\[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_m[x]\ \[Phi]\_n[x]\) 
                \[DifferentialD]x)\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "m", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
where the damping is assumed to be modal. The nonlinear term, in general, \
should be rewritten as\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"NT", "=", 
      RowBox[{
        RowBox[{"(", 
          RowBox[{\(Coefficient[eq723a[\([2]\)], u\_m[t]\^3]\), "/.", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], "->", 
              \(\(\[Phi]\_k\^\[Prime]\)[x]\ 
                \(\[Phi]\_\[ScriptL]\^\[Prime]\)[x]\)}]}], ")"}], " ", 
        \(u\_m[t]\), " ", \(u\_k[t]\), " ", \(u\_\[ScriptL][t]\)}]}]], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"\[Alpha]", " ", "\[Epsilon]", " ", 
      RowBox[{"(", 
        RowBox[{\(\[Integral]\_0\%1\), 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                  MultilineFunction->None], "[", "x", "]"}], " ", 
              RowBox[{
                SubsuperscriptBox["\[Phi]", "\[ScriptL]", "\[Prime]",
                  MultilineFunction->None], "[", "x", "]"}]}], 
            \(\[DifferentialD]x\)}]}], ")"}], " ", 
      RowBox[{"(", 
        RowBox[{\(\[Integral]\_0\%1\), 
          RowBox[{
            RowBox[{\(\[Phi]\_n[x]\), " ", 
              RowBox[{
                SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                  MultilineFunction->None], "[", "x", "]"}]}], 
            \(\[DifferentialD]x\)}]}], ")"}], " ", \(u\_k[t]\), " ", 
      \(u\_m[t]\), " ", \(u\_\[ScriptL][t]\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using the orthonormality of the mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_i[x]\)]],
  " and the results from the corresponding eigenvalue problem, we define the \
following rules:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"notationRule1", "=", 
        RowBox[{"{", 
          RowBox[{
          \(\[Integral]\_0\%1 \[Phi]\_m[x]\ \[Phi]\_n[x] \[DifferentialD]x -> 
              \[Delta]\_\(m, n\)\), ",", 
            RowBox[{
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_n[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], "->", 
              RowBox[{
                RowBox[{"2", "P", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "+", 
                \(\[Omega]\_m\%2\ \[Delta]\_\(m, n\)\)}]}]}], "}"}]}], 
      ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"notationRule2", "=", 
        RowBox[{"{", 
          RowBox[{
          \(\[Integral]\_0\%1 \[Mu][x]\ \[Phi]\_m[x]\ \[Phi]\_n[x] 
                  \[DifferentialD]x -> \[Mu]\_m\ \[Delta]\_\(m, n\)\), ",", 
            RowBox[{
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_n[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], "->", \(-\[Gamma]\_\(n, m\)\)}], 
            ",", 
            RowBox[{
              RowBox[{
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          
                          SubsuperscriptBox["\[Phi]", "\[ScriptL]", 
                            "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}], ")"}], " ", 
                RowBox[{"(", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}], ")"}]}], "->", 
              \(\(-\[Gamma]\_\(k, \[ScriptL]\)\)\ \[Gamma]\_\(n, m\)\)}], 
            ",", \(a_\ \[Delta]\_\(m, n\) :> \((a /. m -> n)\)\)}], "}"}]}], 
      ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and then rewrite ",
  StyleBox["eq723a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(EOM1 = 
      Expand[eq723a[\([1]\)] /. notationRule1] == 
          \((eq723a[\([2]\)] /. u\_m[t]\^3 -> 0)\) + NT //. notationRule2\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(\[Omega]\_n\%2\ u\_n[t]\), "+", 
        RowBox[{
          SubsuperscriptBox["u", "n", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}]}], "==", 
      RowBox[{
      \(\(-4\)\ F\ \[Epsilon]\ Cos[t\ \[CapitalOmega]]\ \[Gamma]\_\(n, m\)\ 
          u\_m[t]\), "-", 
        \(\[Alpha]\ \[Epsilon]\ \[Gamma]\_\(k, \[ScriptL]\)\ 
          \[Gamma]\_\(n, m\)\ u\_k[t]\ u\_m[t]\ u\_\[ScriptL][t]\), "-", 
        RowBox[{"2", " ", "\[Epsilon]", " ", \(\[Mu]\_n\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "n", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`m, k, \)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[ScriptL]\)]],
  " represent summation indicies."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Using the method of multiple scales, we seek a first-order uniform expansion \
in the form\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = 
      u\_n_ -> \((\[Sum]\+\(j = 0\)\%1 \[Epsilon]\^j\ u\_\(n, j\)[#1, #2]\ &)
          \); \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Transforming the total time derivatives in ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into partial derivatives in terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", substituting the ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than \[Epsilon], we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq723b = 
      \((\(EOM1 /. {u\_s_[t] -> u\_s[T\_0, T\_1], 
                  \(\(Derivative[n_]\)[u\_s_]\)[t] -> 
                    \(dt[n]\)[u\_s[T\_0, T\_1]], t -> T\_0}\) /. solRule // 
            ExpandAll)\) /. \[Epsilon]\^\(n_ /; n > 1\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Omega]\_n\%2\ u\_\(n, 0\)[T\_0, T\_1]\), "+", 
        \(\[Epsilon]\ \[Omega]\_n\%2\ u\_\(n, 1\)[T\_0, T\_1]\), "+", 
        RowBox[{"2", " ", "\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["u", \(n, 0\), 
              TagBox[\((1, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], "+", 
        RowBox[{
          SubsuperscriptBox["u", \(n, 0\), 
            TagBox[\((2, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}], "+", 
        RowBox[{"\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["u", \(n, 1\), 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}]}], 
      "==", 
      RowBox[{
      \(\(-4\)\ F\ \[Epsilon]\ Cos[T\_0\ \[CapitalOmega]]\ 
          \[Gamma]\_\(n, m\)\ u\_\(m, 0\)[T\_0, T\_1]\), "-", 
        \(\[Alpha]\ \[Epsilon]\ \[Gamma]\_\(k, \[ScriptL]\)\ 
          \[Gamma]\_\(n, m\)\ u\_\(k, 0\)[T\_0, T\_1]\ 
          u\_\(m, 0\)[T\_0, T\_1]\ u\_\(\[ScriptL], 0\)[T\_0, T\_1]\), "-", 
        RowBox[{"2", " ", "\[Epsilon]", " ", \(\[Mu]\_n\), " ", 
          RowBox[{
            SubsuperscriptBox["u", \(n, 0\), 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}]}]}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq723b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = \(CoefficientList[#, \[Epsilon]]\ &\)\ /@\ eq723b // Thread\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(\[Omega]\_n\%2\ u\_\(n, 0\)[T\_0, T\_1]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", \(n, 0\), 
                TagBox[\((2, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{\(\[Omega]\_n\%2\ u\_\(n, 1\)[T\_0, T\_1]\), "+", 
            RowBox[{"2", " ", 
              RowBox[{
                SubsuperscriptBox["u", \(n, 0\), 
                  TagBox[\((1, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
            "+", 
            RowBox[{
              SubsuperscriptBox["u", \(n, 1\), 
                TagBox[\((2, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
          "==", 
          RowBox[{
          \(\(-4\)\ F\ Cos[T\_0\ \[CapitalOmega]]\ \[Gamma]\_\(n, m\)\ 
              u\_\(m, 0\)[T\_0, T\_1]\), "-", 
            \(\[Alpha]\ \[Gamma]\_\(k, \[ScriptL]\)\ \[Gamma]\_\(n, m\)\ 
              u\_\(k, 0\)[T\_0, T\_1]\ u\_\(m, 0\)[T\_0, T\_1]\ 
              u\_\(\[ScriptL], 0\)[T\_0, T\_1]\), "-", 
            RowBox[{"2", " ", \(\[Mu]\_n\), " ", 
              RowBox[{
                SubsuperscriptBox["u", \(n, 0\), 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(T\_0, T\_1\), 
                "]"}]}]}]}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Because in the presence of damping, all modes that are not directly or \
indirectly excited decay with time, all of the ",
  Cell[BoxData[
      \(TraditionalForm\`u\_\(n, 0\)\)]],
  " decay with time except ",
  Cell[BoxData[
      \(TraditionalForm\`u\_\(1, 0\)\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`u\_\(2, 0\)\)]],
  ". Then it follows from ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol0 = 
      u\_\(i_, 0\) -> 
        Function[{T\_0, T\_1}, 
          A\_i[T\_1]\ Exp[I\ \[Omega]\_i\ T\_0] + 
            A\&_\_i[T\_1]\ Exp[\(-I\)\ \[Omega]\_i\ T\_0]]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["Taking account of the summation indicies, we define", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sumRule = {
        \[Gamma]\_\(k, \[ScriptL]\)\ \[Gamma]\_\(n, m\)\ u\_\(k, 0\)[a__]\ 
            u\_\(m, 0\)[a__]\ u\_\(\[ScriptL], 0\)[a__] -> 
          \[Sum]\+\(k = 1\)\%2
              \(\[Sum]\+\(m = 1\)\%2
                \(\[Sum]\+\(\[ScriptL] = 1\)\%2 \[Gamma]\_\(k, \[ScriptL]\)\ 
                  \[Gamma]\_\(n, m\)\ u\_\(k, 0\)[a]\ u\_\(m, 0\)[a]\ 
                  u\_\(\[ScriptL], 0\)[a]\)\), 
        \[Gamma]\_\(n, m\)\ u\_\(m, 0\)[a__] -> 
          \[Sum]\+\(m = 1\)\%2 \[Gamma]\_\(n, m\)\ u\_\(m, 0\)[a]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Hence, we rewrite ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Clear[order1Eq]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1Eq[n_] = 
      \((eqEps[\([1, 1]\)] /. u\_\(n, 0\) -> u\_\(n, 1\))\) == 
          \((eqEps[\([1, 1]\)] /. u\_\(n, 0\) -> u\_\(n, 1\))\) - 
            \((Subtract\ @@\ eqEps[\([2]\)])\) /. sumRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(\[Omega]\_n\%2\ u\_\(n, 1\)[T\_0, T\_1]\), "+", 
        RowBox[{
          SubsuperscriptBox["u", \(n, 1\), 
            TagBox[\((2, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], "==", 
      RowBox[{
      \(\(-4\)\ F\ Cos[T\_0\ \[CapitalOmega]]\ 
          \((\[Gamma]\_\(n, 1\)\ u\_\(1, 0\)[T\_0, T\_1] + 
              \[Gamma]\_\(n, 2\)\ u\_\(2, 0\)[T\_0, T\_1])\)\), "-", 
        \(\[Alpha]\ 
          \((\[Gamma]\_\(1, 1\)\ \[Gamma]\_\(n, 1\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\^3 + 
              \[Gamma]\_\(1, 2\)\ \[Gamma]\_\(n, 1\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\^2\ u\_\(2, 0\)[T\_0, T\_1] + 
              \[Gamma]\_\(2, 1\)\ \[Gamma]\_\(n, 1\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\^2\ u\_\(2, 0\)[T\_0, T\_1] + 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(n, 2\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\^2\ u\_\(2, 0\)[T\_0, T\_1] + 
              \[Gamma]\_\(2, 2\)\ \[Gamma]\_\(n, 1\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\ u\_\(2, 0\)[T\_0, T\_1]\^2 + 
              \[Gamma]\_\(1, 2\)\ \[Gamma]\_\(n, 2\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\ u\_\(2, 0\)[T\_0, T\_1]\^2 + 
              \[Gamma]\_\(2, 1\)\ \[Gamma]\_\(n, 2\)\ 
                u\_\(1, 0\)[T\_0, T\_1]\ u\_\(2, 0\)[T\_0, T\_1]\^2 + 
              \[Gamma]\_\(2, 2\)\ \[Gamma]\_\(n, 2\)\ 
                u\_\(2, 0\)[T\_0, T\_1]\^3)\)\), "-", 
        RowBox[{"2", " ", \(\[Mu]\_n\), " ", 
          RowBox[{
            SubsuperscriptBox["u", \(n, 0\), 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], "-", 
        RowBox[{"2", " ", 
          RowBox[{
            SubsuperscriptBox["u", \(n, 0\), 
              TagBox[\((1, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}]}]}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol0",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["order1Eq[1]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order1Eq[2]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqOrder1 = \(Array[order1Eq, 2] /. sol0 // TrigToExp\) // ExpandAll; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["Principal Parametric Resonance of the First Mode", "Subsubsection",
  CellFrame->False,
  CellDingbat->"\[FilledSquare]",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_2}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma]\_2 + 2\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] 
          1\/3\ \((\(-\[Epsilon]\)\ \[Sigma]\_1 + \[Omega]\_2)\), 
        \[CapitalOmega] \[Rule] 
          1\/3\ \((
              \(-2\)\ \[Epsilon]\ \[Sigma]\_1 + 3\ \[Epsilon]\ \[Sigma]\_2 + 
                2\ \[Omega]\_2)\)}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand sides of ",
  StyleBox["eqOrder1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the solvability condition as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[Coefficient[\(-eqOrder1[\([i, 2]\)]\) /. expRule[i], 
            Exp[I\ \[Omega]\_i\ T\_0]] == 0, {i, 2}]; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "+", 
            \(2\ I\ A\_1\ \[Mu]\_1\ \[Omega]\_1\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ F\ A\&_\_1\ 
              \[Gamma]\_\(1, 1\)\), "+", 
            \(3\ \[Alpha]\ A\_1\%2\ A\&_\_1\ \[Gamma]\_\(1, 1\)\%2\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2\)\ F\ A\_2\ 
              \[Gamma]\_\(1, 2\)\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_2\ A\&_\_1\%2\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(1, 2\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 2\)\%2\), "+", 
            \(E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_2\ A\&_\_1\%2\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 2\)\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 1\)\ 
              \[Gamma]\_\(2, 2\)\)}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "+", 
            \(2\ I\ A\_2\ \[Mu]\_2\ \[Omega]\_2\), "+", 
            \(2\ E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2\)\ F\ 
              A\_1\ \[Gamma]\_\(2, 1\)\), "+", 
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_1\%3\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(1, 2\)\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(2, 1\)\%2\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(1, 1\)\ 
              \[Gamma]\_\(2, 2\)\), "+", 
            \(3\ \[Alpha]\ A\_2\%2\ A\&_\_2\ \[Gamma]\_\(2, 2\)\%2\)}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which is in agreement with that obtained by direct approach.", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Principal Parametric Resonance of the Second Mode", "Subsubsection",
  CellFrame->False,
  CellDingbat->"\[FilledSquare]",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\ \[Sigma]\_2}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
        \[CapitalOmega] \[Rule] 
          2\ \[Epsilon]\ \[Sigma]\_1 + \[Epsilon]\ \[Sigma]\_2 + 
            6\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] 
          1\/3\ \((\(-\[Epsilon]\)\ \[Sigma]\_1 + \[Omega]\_2)\), 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma]\_2 + 2\ \[Omega]\_2}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand sides of ",
  StyleBox["eqOrder1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the solvability condition as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[Coefficient[\(-eqOrder1[\([i, 2]\)]\) /. expRule[i], 
            Exp[I\ \[Omega]\_i\ T\_0]] == 0, {i, 2}]; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "+", 
            \(2\ I\ A\_1\ \[Mu]\_1\ \[Omega]\_1\), "+", 
            \(3\ \[Alpha]\ A\_1\%2\ A\&_\_1\ \[Gamma]\_\(1, 1\)\%2\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_2\ A\&_\_1\%2\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(1, 2\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 2\)\%2\), "+", 
            \(E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_2\ A\&_\_1\%2\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 2\)\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 1\)\ 
              \[Gamma]\_\(2, 2\)\)}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "+", 
            \(2\ I\ A\_2\ \[Mu]\_2\ \[Omega]\_2\), "+", 
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_1\%3\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(1, 2\)\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(2, 1\)\%2\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ F\ A\&_\_2\ 
              \[Gamma]\_\(2, 2\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(1, 1\)\ 
              \[Gamma]\_\(2, 2\)\), "+", 
            \(3\ \[Alpha]\ A\_2\%2\ A\&_\_2\ \[Gamma]\_\(2, 2\)\%2\)}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which is in agreement with that obtained by direct approach.", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Combination Parametric Resonance of the Two Modes", "Subsubsection",
  CellFrame->False,
  CellDingbat->"\[FilledSquare]",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
        \[CapitalOmega] == 
          \[Omega]\_1 + \[Omega]\_2 + \[Epsilon]\ \[Sigma]\_2}; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
        \[CapitalOmega] \[Rule] 
          \[Epsilon]\ \[Sigma]\_1 + \[Epsilon]\ \[Sigma]\_2 + 
            4\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] 
          1\/3\ \((\(-\[Epsilon]\)\ \[Sigma]\_1 + \[Omega]\_2)\), 
        \[CapitalOmega] \[Rule] 
          1\/3\ \((
              \(-\[Epsilon]\)\ \[Sigma]\_1 + 3\ \[Epsilon]\ \[Sigma]\_2 + 
                4\ \[Omega]\_2)\)}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand sides of ",
  StyleBox["eqOrder1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the solvability condition as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[Coefficient[\(-eqOrder1[\([i, 2]\)]\) /. expRule[i], 
            Exp[I\ \[Omega]\_i\ T\_0]] == 0, {i, 2}]; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "+", 
            \(2\ I\ A\_1\ \[Mu]\_1\ \[Omega]\_1\), "+", 
            \(3\ \[Alpha]\ A\_1\%2\ A\&_\_1\ \[Gamma]\_\(1, 1\)\%2\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ F\ A\&_\_2\ 
              \[Gamma]\_\(1, 2\)\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_2\ A\&_\_1\%2\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(1, 2\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 2\)\%2\), "+", 
            \(E\^\(I\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_2\ A\&_\_1\%2\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 2\)\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\ \[Gamma]\_\(1, 1\)\ 
              \[Gamma]\_\(2, 2\)\)}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "+", 
            \(2\ I\ A\_2\ \[Mu]\_2\ \[Omega]\_2\), "+", 
            \(2\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ F\ A\&_\_1\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\ \[Alpha]\ A\_1\%3\ 
              \[Gamma]\_\(1, 1\)\ \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(1, 2\)\ 
              \[Gamma]\_\(2, 1\)\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(2, 1\)\%2\), "+", 
            \(2\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\ \[Gamma]\_\(1, 1\)\ 
              \[Gamma]\_\(2, 2\)\), "+", 
            \(3\ \[Alpha]\ A\_2\%2\ A\&_\_2\ \[Gamma]\_\(2, 2\)\%2\)}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which is in agreement with that obtained by direct approach.", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.2.4 Method of Time-Averaged Lagrangian", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
As a second alternative, we derive the modulation equations by using the \
method of time-averaged Lagrangian. The nondimensional Lagrangian of the beam \
can be expressed as (Nayfeh, 1998)\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(Lagr1 = 
      \(1\/2\) \(\[Integral]\_0\%1\(\((\[PartialD]\_t w[x, t])\)\^2\) 
              \[DifferentialD]x\) - 
        \(1\/2\) 
          \(\[Integral]\_0\%1\(\((\[PartialD]\_\(x, x\)w[x, t])\)\^2\) 
              \[DifferentialD]x\) - 
        \((P + 2  \[Epsilon]\ F\ Cos[\[CapitalOmega]\ t])\) 
          \(\[Integral]\_0\%1\(\((\[PartialD]\_x w[x, t])\)\^2\) 
              \[DifferentialD]x\) - 
        \(1\/4\) \[Epsilon]\ \[Alpha] 
          \((\[Integral]\_0\%1\(\((\[PartialD]\_x w[x, t])\)\^2\) 
                  \[DifferentialD]x)\)\^2 + C[t]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`C[t]\)]],
  " is independent of ",
  Cell[BoxData[
      \(TraditionalForm\`w\)]],
  ". Transforming the total time derivatives into partial derivatives in \
terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", we modify ",
  StyleBox["Lagr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Lagr2 = 
      \(Lagr1 /. Integrate -> int\) /. {w[x_, t] -> w[x, T\_0, T\_1], 
          \(\(Derivative[m_, n_]\)[w]\)[x_, t] -> 
            \(dt[n]\)[D[w[x, T\_0, T\_1], {x, m}]], t -> T\_0}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{\(C[T\_0]\), "+", 
      RowBox[{\(1\/2\), " ", 
        RowBox[{"int", "[", 
          RowBox[{
            SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  RowBox[{"\[Epsilon]", " ", 
                    RowBox[{
                      SuperscriptBox["w", 
                        TagBox[\((0, 0, 1)\),
                          Derivative],
                        MultilineFunction->None], "[", \(x, T\_0, T\_1\), 
                      "]"}]}], "+", 
                  RowBox[{
                    SuperscriptBox["w", 
                      TagBox[\((0, 1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, T\_0, T\_1\), 
                    "]"}]}], ")"}], "2"], ",", \({x, 0, 1}\)}], "]"}]}], 
      "-", 
      RowBox[{\((P + 2\ F\ \[Epsilon]\ Cos[T\_0\ \[CapitalOmega]])\), " ", 
        RowBox[{"int", "[", 
          RowBox[{
            SuperscriptBox[
              RowBox[{
                SuperscriptBox["w", 
                  TagBox[\((1, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, T\_0, T\_1\), "]"}], 
              "2"], ",", \({x, 0, 1}\)}], "]"}]}], "-", 
      RowBox[{\(1\/4\), " ", "\[Alpha]", " ", "\[Epsilon]", " ", 
        SuperscriptBox[
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SuperscriptBox["w", 
                    TagBox[\((1, 0, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(x, T\_0, T\_1\), "]"}], 
                "2"], ",", \({x, 0, 1}\)}], "]"}], "2"]}], "-", 
      RowBox[{\(1\/2\), " ", 
        RowBox[{"int", "[", 
          RowBox[{
            SuperscriptBox[
              RowBox[{
                SuperscriptBox["w", 
                  TagBox[\((2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, T\_0, T\_1\), "]"}], 
              "2"], ",", \({x, 0, 1}\)}], "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["To this end, we let", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(solRule = 
      w -> \((Evaluate[
              Sum[A\_i[#3]\ \[Phi]\_i[#1]\ E\^\(I\ \[Omega]\_i\ #2\) + 
                  A\&_\_i[#3]\ \[Phi]\_i[#1]\ 
                    E\^\(\(-I\)\ \[Omega]\_i\ #2\), {i, 2}]]\ &)\)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(w \[Rule] 
      \((E\^\(I\ #2\ \[Omega]\_1\)\ A\_1[#3]\ \[Phi]\_1[#1] + 
            E\^\(I\ #2\ \[Omega]\_2\)\ A\_2[#3]\ \[Phi]\_2[#1] + 
            E\^\(\(-I\)\ #2\ \[Omega]\_1\)\ \[Phi]\_1[#1]\ A\&_\_1[#3] + 
            E\^\(\(-I\)\ #2\ \[Omega]\_2\)\ \[Phi]\_2[#1]\ A\&_\_2[#3]&)
        \)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the first independent variable (#1) stands for ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and the last two independent variables (#2 and #3) stand for the two time \
scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ". Substituting ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["Lagr2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to simplify the expansions of the integrands, and collecting the \
coefficients of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(order1Lagr = 
      \(\(\(\(Lagr2 /. solRule\) /. intRule1\) //. intRule2 // TrigToExp\) // 
          Expand\) // Coefficient[#, \[Epsilon]]\ &; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["Principal Parametric Resonance of the First Mode", "Subsubsection",
  CellFrame->False,
  CellDingbat->"\[FilledSquare]",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ResonanceConds = {
          \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
          \[CapitalOmega] == 2  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_2} /. 
        Equal -> Rule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
      \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma]\_2 + 2\ \[Omega]\_1}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the slowly varying terms from ",
  StyleBox["order1Lagr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(TAL = 
      \(\(order1Lagr /. 
            Exp[a_] :> 
              Exp[Expand[a /. ResonanceConds] /. \[Epsilon]\ T\_0 -> T\_1]
                \) /. Exp[_\ T\_0 + _. ] -> 0\) /. int -> Integrate\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(-E\^\(\(-I\)\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^2\)}], 
      "-", 
      RowBox[{
      "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2\)\), " ", 
        "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
        \(A\&_\_1[T\_1]\)}], "-", 
      RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1[T\_1]\^2\)}], 
      "-", 
      RowBox[{\(3\/2\), " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
        \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\^2\)}], "-", 
      RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
        \(A\&_\_1[T\_1]\^3\)}], "-", 
      RowBox[{
      "2", " ", \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2\)\), 
        " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{\(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^3\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"4", " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", \(A\_1[T\_1]\), 
        " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"2", " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
        \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
      "-", 
      RowBox[{\(3\/2\), " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
        \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\^2\)}], "-", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_1\), " ", \(A\&_\_1[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "1", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_2\), " ", \(A\&_\_2[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "2", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_1\), " ", \(A\_1[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_2\), " ", \(A\_2[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The Euler-Lagrange equations corresponding to the ",
  StyleBox["TAL",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod1", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"D", "[", 
            RowBox[{
              RowBox[{"D", "[", 
                RowBox[{"TAL", ",", 
                  RowBox[{
                    SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                "]"}], ",", \(T\_1\)}], "]"}], "-", 
          \(D[TAL, A\&_\_1[T\_1]]\)}], "==", "0"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{
        "2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1 - I\ T\_1\ \[Sigma]\_2\)\), 
          " ", "F", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1[T\_1]\)}], 
        "+", 
        RowBox[{"3", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{
        "3", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
          \(A\&_\_1[T\_1]\^2\)}], "+", 
        RowBox[{"4", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
          \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", 
          \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
          \(\[Omega]\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "1", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod2", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"D", "[", 
            RowBox[{
              RowBox[{"D", "[", 
                RowBox[{"TAL", ",", 
                  RowBox[{
                    SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                "]"}], ",", \(T\_1\)}], "]"}], "-", 
          \(D[TAL, A\&_\_2[T\_1]]\)}], "==", "0"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{
        "2", " ", 
          \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1 + I\ T\_1\ \[Sigma]\_2\)\), " ", 
          "F", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\)}], 
        "+", 
        RowBox[{\(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^3\)}], 
        "+", 
        RowBox[{"4", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
          \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"3", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", 
          \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
          \(\[Omega]\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "2", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Adding linear viscous damping to ",
  StyleBox["eqMod1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eqMod2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and performing integration by parts yields the same modulation equations \
as obtained by directly attacking the partial-differential system."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Principal Parametric Resonance of the Second Mode", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ResonanceConds = 
      \(Solve[{\[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
            \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\ \[Sigma]\_2}, {
            \[Omega]\_2, \[CapitalOmega]}]\)[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
      \[CapitalOmega] \[Rule] 
        2\ \[Epsilon]\ \[Sigma]\_1 + \[Epsilon]\ \[Sigma]\_2 + 6\ \[Omega]\_1}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the slowly varying terms from ",
  StyleBox["order1Lagr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(TAL = 
      \(\(order1Lagr /. 
            Exp[a_] :> 
              Exp[Expand[a /. ResonanceConds] /. \[Epsilon]\ T\_0 -> T\_1]
                \) /. Exp[_\ T\_0 + _. ] -> 0\) /. int -> Integrate\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(-E\^\(\(-I\)\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\^2\)}], 
      "-", 
      RowBox[{\(3\/2\), " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
        \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\^2\)}], "-", 
      RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
        \(A\&_\_1[T\_1]\^3\)}], "-", 
      RowBox[{\(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^3\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"4", " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", \(A\_1[T\_1]\), 
        " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"2", " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
        \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
      "-", 
      RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_2[T\_1]\^2\)}], 
      "-", 
      RowBox[{\(3\/2\), " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
        \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\^2\)}], "-", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_1\), " ", \(A\&_\_1[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "1", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_2\), " ", \(A\&_\_2[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "2", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_1\), " ", \(A\_1[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_2\), " ", \(A\_2[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The Euler-Lagrange equations corresponding to the ",
  StyleBox["TAL",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod1", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"D", "[", 
            RowBox[{
              RowBox[{"D", "[", 
                RowBox[{"TAL", ",", 
                  RowBox[{
                    SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                "]"}], ",", \(T\_1\)}], "]"}], "-", 
          \(D[TAL, A\&_\_1[T\_1]]\)}], "==", "0"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"3", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{
        "3", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
          \(A\&_\_1[T\_1]\^2\)}], "+", 
        RowBox[{"4", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
          \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", 
          \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
          \(\[Omega]\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "1", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod2", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"D", "[", 
            RowBox[{
              RowBox[{"D", "[", 
                RowBox[{"TAL", ",", 
                  RowBox[{
                    SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                "]"}], ",", \(T\_1\)}], "]"}], "-", 
          \(D[TAL, A\&_\_2[T\_1]]\)}], "==", "0"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^3\)}], 
        "+", 
        RowBox[{"4", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
          \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_2[T\_1]\)}], 
        "+", 
        RowBox[{"3", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", 
          \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
          \(\[Omega]\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "2", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Adding linear viscous damping to ",
  StyleBox["eqMod1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eqMod2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and performing integration by parts yields the same modulation equations \
as obtained by directly attacking the partial-differential system."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Combination Parametric Resonance of the Two Modes", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ResonanceConds = 
      \(Solve[{\[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\ \[Sigma]\_1, 
            \[CapitalOmega] == 
              \[Omega]\_1 + \[Omega]\_2 + \[Epsilon]\ \[Sigma]\_2}, {
            \[Omega]\_2, \[CapitalOmega]}]\)[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] \[Epsilon]\ \[Sigma]\_1 + 3\ \[Omega]\_1, 
      \[CapitalOmega] \[Rule] 
        \[Epsilon]\ \[Sigma]\_1 + \[Epsilon]\ \[Sigma]\_2 + 4\ \[Omega]\_1}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the slowly varying terms from ",
  StyleBox["order1Lagr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(TAL = 
      \(\(order1Lagr /. 
            Exp[a_] :> 
              Exp[Expand[a /. ResonanceConds] /. \[Epsilon]\ T\_0 -> T\_1]
                \) /. Exp[_\ T\_0 + _. ] -> 0\) /. int -> Integrate\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(-2\), " ", \(E\^\(\(-I\)\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
        \(A\_2[T\_1]\)}], "-", 
      RowBox[{\(3\/2\), " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
        \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\^2\)}], "-", 
      RowBox[{\(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
        \(A\&_\_1[T\_1]\^3\)}], "-", 
      RowBox[{\(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^3\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1[T\_1]\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"4", " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", \(A\_1[T\_1]\), 
        " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\), " ", 
        \(A\&_\_2[T\_1]\)}], "-", 
      RowBox[{"2", " ", "\[Alpha]", " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", 
        RowBox[{"(", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
        \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
      "-", 
      RowBox[{\(3\/2\), " ", "\[Alpha]", " ", 
        SuperscriptBox[
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
        \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\^2\)}], "-", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_1\), " ", \(A\&_\_1[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "1", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "-", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_2\), " ", \(A\&_\_2[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "2", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_1\), " ", \(A\_1[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}], "+", 
      RowBox[{
      "I", " ", 
        \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
        \(\[Omega]\_2\), " ", \(A\_2[T\_1]\), " ", 
        RowBox[{
          SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
            MultilineFunction->None], "[", \(T\_1\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The Euler-Lagrange equations corresponding to the ",
  StyleBox["TAL",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod1", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"D", "[", 
            RowBox[{
              RowBox[{"D", "[", 
                RowBox[{"TAL", ",", 
                  RowBox[{
                    SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                "]"}], ",", \(T\_1\)}], "]"}], "-", 
          \(D[TAL, A\&_\_1[T\_1]]\)}], "==", "0"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"3", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{
        "3", " ", \(E\^\(I\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_2[T\_1]\), " ", 
          \(A\&_\_1[T\_1]\^2\)}], "+", 
        RowBox[{"2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_2[T\_1]\)}], 
        "+", 
        RowBox[{"4", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
          \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", 
          \((\[Integral]\_0\%1\( \[Phi]\_1[x]\^2\) \[DifferentialD]x)\), " ", 
          \(\[Omega]\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "1", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod2", "=", 
      RowBox[{
        RowBox[{
          RowBox[{"D", "[", 
            RowBox[{
              RowBox[{"D", "[", 
                RowBox[{"TAL", ",", 
                  RowBox[{
                    SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                "]"}], ",", \(T\_1\)}], "]"}], "-", 
          \(D[TAL, A\&_\_2[T\_1]]\)}], "==", "0"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(E\^\(\(-I\)\ T\_1\ \[Sigma]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\^3\)}], 
        "+", 
        RowBox[{"2", " ", \(E\^\(I\ T\_1\ \[Sigma]\_2\)\), " ", "F", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\&_\_1[T\_1]\)}], 
        "+", 
        RowBox[{"4", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], 
        "+", 
        RowBox[{"2", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", \(A\_1[T\_1]\), " ", 
          \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"3", " ", "\[Alpha]", " ", 
          SuperscriptBox[
            RowBox[{"(", 
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "2", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], 
                  \(\[DifferentialD]x\)}]}], ")"}], "2"], " ", 
          \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", 
          \((\[Integral]\_0\%1\( \[Phi]\_2[x]\^2\) \[DifferentialD]x)\), " ", 
          \(\[Omega]\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "2", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Adding linear viscous damping to ",
  StyleBox["eqMod1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eqMod2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and performing integration by parts yields the same modulation equations \
as obtained by directly attacking the partial-differential system."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.3 Cantilever Beam", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[TextData[StyleBox["7.3.1 EOM and BC's",
  FontWeight->"Bold"]], "Subsection",
  TextAlignment->Left,
  TextJustification->1,
  Background->None],

Cell[TextData[{
  "We consider the nonlinear nonplanar response of an inextensional \
cantilever beam to an external excitation of one of its flexural modes. The \
lowest torsional frequencies of the beam considered are much higher than the \
frequencies of the excited flexural modes so that the torsional inertia can \
be neglected. We assume that the beam is uniform and homogeneous. The \
transverse deflections ",
  Cell[BoxData[
      \(TraditionalForm\`v[x, t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`w[x, t]\)]],
  " of the beam at the position ",
  StyleBox["x",
    FontSlant->"Italic"],
  " and time ",
  StyleBox["t",
    FontSlant->"Italic"],
  " are governed by the nondimensional integral-partial-differential equation \
(Crespo da Silva and Glynn, 1978)"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(EOM1 = {
        \[PartialD]\_\(t, t\)v[x, t] + 
            \[Beta]\_y\ \[PartialD]\_\(x, x, x, x\)v[x, t] == 
          \(-2\) \[Epsilon]\ \[Mu][x]\ \[PartialD]\_t v[x, t] + 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\ 
              \[PartialD]\_x\((
                  \[PartialD]\_\(x, x\)w[x, t]\ 
                      \(\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                          \[PartialD]\_\(x, x\)w[x, t] \[DifferentialD]x\) - 
                    \[PartialD]\_\(x, x, x\)w[x, t]\ 
                      \(\[Integral]\_0\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                          \[PartialD]\_x w[x, t] \[DifferentialD]x\))\) - 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\^2\/\[Beta]\_\[Gamma]\ 
              \[PartialD]\_\(x, x\)\((
                  \[PartialD]\_\(x, x\)w[x, t]\ 
                    \(\[Integral]\_0\%x
                        \(\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                          \[PartialD]\_\(x, x\)w[x, t] \[DifferentialD]x\ 
                          \[DifferentialD]x\)\))\) - 
            \[Epsilon]\ \[Beta]\_y\ 
              \[PartialD]\_x\((
                  \[PartialD]\_x v[x, t]\ 
                    \[PartialD]\_x\((
                        \[PartialD]\_x v[x, t]\ 
                            \[PartialD]\_\(x, x\)v[x, t] + 
                          \[PartialD]\_x w[x, t]\ 
                            \[PartialD]\_\(x, x\)w[x, t])\))\) - 
            \[Epsilon]\ \(1\/2\) 
              \[PartialD]\_x\((
                  \[PartialD]\_x v[x, t]\ 
                    \(\[Integral]\_1\%x
                        \[PartialD]\_\(t, t\)\((
                            \[Integral]\_0\%x
                                \((\((\[PartialD]\_x v[x, t])\)\^2 + 
                                    \((\[PartialD]\_x w[x, t])\)\^2)\) 
                                \[DifferentialD]x)\) \[DifferentialD]x\))\) - 
            \[Epsilon]\ F[x]\ Cos[\[CapitalOmega]\ t], 
        \[PartialD]\_\(t, t\)w[x, t] + \[PartialD]\_\(x, x, x, x\)w[x, t] == 
          \(-2\) \[Epsilon]\ \[Mu][x]\ \[PartialD]\_t w[x, t] - 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\ 
              \[PartialD]\_x\((
                  \[PartialD]\_\(x, x\)v[x, t]\ 
                      \(\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                          \[PartialD]\_\(x, x\)w[x, t] \[DifferentialD]x\) - 
                    \[PartialD]\_\(x, x, x\)v[x, t]\ 
                      \(\[Integral]\_0\%x\[PartialD]\_\(x, x\)w[x, t]\ 
                          \[PartialD]\_x v[x, t] \[DifferentialD]x\))\) - 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\^2\/\[Beta]\_\[Gamma]\ 
              \[PartialD]\_\(x, x\)\((
                  \[PartialD]\_\(x, x\)v[x, t]\ 
                    \(\[Integral]\_0\%x
                        \(\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                          \[PartialD]\_\(x, x\)w[x, t] \[DifferentialD]x\ 
                          \[DifferentialD]x\)\))\) - 
            \[Epsilon]\ 
              \[PartialD]\_x\((
                  \[PartialD]\_x w[x, t]\ 
                    \[PartialD]\_x\((
                        \[PartialD]\_x v[x, t]\ 
                            \[PartialD]\_\(x, x\)v[x, t] + 
                          \[PartialD]\_x w[x, t]\ 
                            \[PartialD]\_\(x, x\)w[x, t])\))\) - 
            \[Epsilon]\ \(1\/2\) 
              \[PartialD]\_x\((
                  \[PartialD]\_x w[x, t]\ 
                    \(\[Integral]\_1\%x
                        \[PartialD]\_\(t, t\)\((
                            \[Integral]\_0\%x
                                \((\((\[PartialD]\_x v[x, t])\)\^2 + 
                                    \((\[PartialD]\_x w[x, t])\)\^2)\) 
                                \[DifferentialD]x)\) \[DifferentialD]x\))\)}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["along with the homogeneous boundary conditions", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC1 = {v[0, t] == 0, \[PartialD]\_x v[x, t] == 0 /. x -> 0, 
        \[PartialD]\_\(x, x\)v[x, t] == 0 /. x -> 1, 
        \[PartialD]\_\(x, x, x\)v[x, t] == 0 /. x -> 1}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(v[0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["v", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, t\), "]"}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["v", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["v", 
              TagBox[\((3, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC = Join[BC1, BC1 /. v -> w]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(v[0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["v", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, t\), "]"}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["v", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["v", 
              TagBox[\((3, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}], 
        ",", \(w[0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, t\), "]"}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((3, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The spatial derivatives outside the integrals on the right-hand sides of ",
  
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " result in lengthy expressions. To simplify the symbolic computation, we \
first define some operators as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(op1[v_, w_] = 
      \[PartialD]\_\(x, x\)w[x, t]\ 
            \(\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                \[PartialD]\_\(x, x\)w[x, t] \[DifferentialD]x\) - 
          \[PartialD]\_\(x, x, x\)w[x, t]\ 
            \(\[Integral]\_0\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                \[PartialD]\_x w[x, t] \[DifferentialD]x\) /. 
        Integrate -> int[1]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(int[1]\), "[", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox["v", 
                  TagBox[\((2, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, t\), "]"}], " ", 
              RowBox[{
                SuperscriptBox["w", 
                  TagBox[\((2, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, t\), "]"}]}], ",", 
            \({x, 1, x}\)}], "]"}], " ", 
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((2, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}]}], "-", 
      RowBox[{
        RowBox[{\(int[1]\), "[", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox["w", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, t\), "]"}], " ", 
              RowBox[{
                SuperscriptBox["v", 
                  TagBox[\((2, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, t\), "]"}]}], ",", 
            \({x, 0, x}\)}], "]"}], " ", 
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((3, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(op2[v_, w_] = 
      Hold[\[PartialD]\_\(x, x\)w[x, t]\ 
              \(\[Integral]\_0\%x
                  \(\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                    \[PartialD]\_\(x, x\)w[x, t] \[DifferentialD]x\ 
                    \[DifferentialD]x\)\)] /. Integrate -> int[2] // 
        ReleaseHold\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(int[2]\), "[", 
        RowBox[{
          RowBox[{
            RowBox[{
              SuperscriptBox["v", 
                TagBox[\((2, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(x, t\), "]"}], " ", 
            RowBox[{
              SuperscriptBox["w", 
                TagBox[\((2, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(x, t\), "]"}]}], ",", 
          \({x, 0, x}\), ",", \({x, 1, x}\)}], "]"}], " ", 
      RowBox[{
        SuperscriptBox["w", 
          TagBox[\((2, 0)\),
            Derivative],
          MultilineFunction->None], "[", \(x, t\), "]"}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(op3[v_, w_] = 
      \[PartialD]\_x v[x, t]\ 
          \(\[Integral]\_1\%x
              \[PartialD]\_\(t, t\)\((
                  \[Integral]\_0\%x
                      \((\((\[PartialD]\_x v[x, t])\)\^2 + 
                          \((\[PartialD]\_x w[x, t])\)\^2)\) 
                      \[DifferentialD]x)\) \[DifferentialD]x\) /. 
        Integrate -> int[1]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(int[1]\), "[", 
        RowBox[{
          RowBox[{\(int[1]\), "[", 
            RowBox[{
              RowBox[{
                RowBox[{"2", " ", 
                  SuperscriptBox[
                    RowBox[{
                      SuperscriptBox["v", 
                        TagBox[\((1, 1)\),
                          Derivative],
                        MultilineFunction->None], "[", \(x, t\), "]"}], 
                    "2"]}], "+", 
                RowBox[{"2", " ", 
                  SuperscriptBox[
                    RowBox[{
                      SuperscriptBox["w", 
                        TagBox[\((1, 1)\),
                          Derivative],
                        MultilineFunction->None], "[", \(x, t\), "]"}], 
                    "2"]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{
                    SuperscriptBox["v", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}], " ", 
                  RowBox[{
                    SuperscriptBox["v", 
                      TagBox[\((1, 2)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{
                    SuperscriptBox["w", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}], " ", 
                  RowBox[{
                    SuperscriptBox["w", 
                      TagBox[\((1, 2)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}]}]}], 
              ",", \({x, 0, x}\)}], "]"}], ",", \({x, 1, x}\)}], "]"}], " ", 
      RowBox[{
        SuperscriptBox["v", 
          TagBox[\((1, 0)\),
            Derivative],
          MultilineFunction->None], "[", \(x, t\), "]"}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`int[i]\)]],
  " indicates the whole expression, including the integrand and the term \
multiplied by ",
  Cell[BoxData[
      \(TraditionalForm\`\(int[i]\)\ [expr]\)]],
  ", will later be wrapped by the ",
  Cell[BoxData[
      \(TraditionalForm\`i\)]],
  "th spatial derivative in the solvability condition."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Then we rewrite ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(EOM = {
        \[PartialD]\_\(t, t\)v[x, t] + 
            \[Beta]\_y\ \[PartialD]\_\(x, x, x, x\)v[x, t] == 
          \(-2\) \[Epsilon]\ \[Mu][x]\ \[PartialD]\_t v[x, t] + 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\ op1[v, w] - 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\^2\/\[Beta]\_\[Gamma]\ 
              op2[v, w] - 
            \[Epsilon]\ \[Beta]\_y\ 
              \[PartialD]\_x\((
                  \[PartialD]\_x v[x, t]\ 
                    \[PartialD]\_x\((
                        \[PartialD]\_x v[x, t]\ 
                            \[PartialD]\_\(x, x\)v[x, t] + 
                          \[PartialD]\_x w[x, t]\ 
                            \[PartialD]\_\(x, x\)w[x, t])\))\) - 
            \[Epsilon]\ \(1\/2\) op3[v, w] - 
            \[Epsilon]\ F[x]\ Cos[\[CapitalOmega]\ t], 
        \[PartialD]\_\(t, t\)w[x, t] + \[PartialD]\_\(x, x, x, x\)w[x, t] == 
          \(-2\) \[Epsilon]\ \[Mu][x]\ \[PartialD]\_t w[x, t] - 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\ op1[w, v] - 
            \[Epsilon]\ \((1 - \[Beta]\_y)\)\^2\/\[Beta]\_\[Gamma]\ 
              op2[w, v] - 
            \[Epsilon]\ 
              \[PartialD]\_x\((
                  \[PartialD]\_x w[x, t]\ 
                    \[PartialD]\_x\((
                        \[PartialD]\_x v[x, t]\ 
                            \[PartialD]\_\(x, x\)v[x, t] + 
                          \[PartialD]\_x w[x, t]\ 
                            \[PartialD]\_\(x, x\)w[x, t])\))\) - 
            \[Epsilon]\ \(1\/2\) op3[w, v]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "It follows from ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the linear undamped natural frequencies and their corresponding mode \
shapes are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\({\[Omega]\&^\_\(1, m\) == \[Lambda]\_m\%2\ \@\[Beta]\_y, 
      \[Omega]\_\(2, n\) == \[Lambda]\_n\%2, \n
      \[Phi]\_i[x] == 
        Cosh[\[Lambda]\_i\ x] - Cos[\[Lambda]\_i\ x] + 
          \(Cos[\[Lambda]\_i] + Cosh[\[Lambda]\_i]\)\/\(Sin[\[Lambda]\_i] + 
                Sinh[\[Lambda]\_i]\)\ 
            \((Sin[\[Lambda]\_i\ x] - Sinh[\[Lambda]\_i\ x])\)\  (*\ i = m, n
        \ *) }; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\&^\_\(1, \ m\)\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_\(2, \ n\)\)]],
  " are the natural frequencies in the ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`z\)]],
  " directions and the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_i\)]],
  " are the roots of"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[Cell[BoxData[
    \(1 + Cos[\[Lambda]\_i]\ Cosh[\[Lambda]\_i] == 0\)], "Input"]], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We consider the case of one-to-one internal resonance between the ",
  Cell[BoxData[
      \(TraditionalForm\`m\)]],
  "th mode in the ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " or ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  " direction and the ",
  Cell[BoxData[
      \(TraditionalForm\`n\)]],
  "th mode in the ",
  Cell[BoxData[
      \(TraditionalForm\`z\)]],
  " or ",
  Cell[BoxData[
      \(TraditionalForm\`w\)]],
  " direction; that is, ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[Omega]\&^\_\(1, \ m\) \[TildeTilde] \[Omega]\_\(2, \ n\)\)]],
  ". To express the nearness of these frequencies quantitatively, we let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(betayRule = {
        \[Beta]\_y -> 1 + \[Delta]\_0 + \[Epsilon]\ \[Delta]\_1}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "so that ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[Omega]\_\(1, \ m\) = 
        \(\(\[Lambda]\_m\%2\) \@\(1 + \[Delta]\_0\) = 
          \(\[Lambda]\_n\%2 = \[Omega]\_\(2, \ n\)\)\)\)]],
  ". The beam has a near-square cross-section when ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\_0 = 0\)]],
  ". We assume neither of these two modes is involved in an internal \
resonance with any other mode. Moreover, we consider a primary resonance of \
the flexural mode in the ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " direction."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To manipulate some complicated integrals, we define the following rules:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(intRule3 = {
        \(int[i_]\)[fun_, arg2__] :> \(int[i]\)[Expand[fun], arg2]}; \n
    intRule4 = {
        \(int[i_]\)[a_ + b_, arg2__] :> 
          \(int[i]\)[a, arg2] + \(int[i]\)[b, arg2], 
        \(int[i_]\)[\[Epsilon]\^n_. \ fun_, arg2__] :> 
          \[Epsilon]\^n\ \(int[i]\)[fun, arg2], 
        \(int[i_]\)[a_\ fun_, a1_, b1___] :> 
          a\ \(int[i]\)[fun, a1, b1] /; FreeQ[a, First[a1]], 
        \(int[i_]\)[\(int[i_]\)[a1_, a2_]\ fun_. , a3__] :> 
          \(int[i]\)[a1, a2]\ \(int[i]\)[fun, a3] /; 
            FreeQ[Rest[a2], First[a2]], 
        \(int[i_]\)[\(int[i_]\)[a1_, a2_], a3__] :> \(int[i]\)[a1, a3, a2]}; 
    \)], "Input",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.3.2 Direct Attack of the Continuous Problem", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this section, we directly attack the integral-partial-differential \
equation ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and associated boundary conditions ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and seek a first-order uniform expansion of their solution in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(solRule = {
        v -> \((Evaluate[Sum[\[Epsilon]\^j\ v\_j[#1, #2, #3], {j, 0, 1}]]\ &)
            \), w -> 
          \((Evaluate[Sum[\[Epsilon]\^j\ w\_j[#1, #2, #3], {j, 0, 1}]]\ &)\)}
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({v \[Rule] \((v\_0[#1, #2, #3] + \[Epsilon]\ v\_1[#1, #2, #3]&)\), 
      w \[Rule] \((w\_0[#1, #2, #3] + \[Epsilon]\ w\_1[#1, #2, #3]&)\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the first independent variable stands for ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and the last two independent variables stand for the two time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ". Transforming the total time derivatives in ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into partial derivatives in terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", substituting the ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["betayRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], discarding terms of order \
higher than \[Epsilon], and using ",
  StyleBox["intRule3",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule4",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to simplify the expansions of the integrands, we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((eq732a = 
          \((\(\((\(\(Join[EOM, BC] /. {v[x_, t] -> v[x, T\_0, T\_1], 
                                w[x_, t] -> w[x, T\_0, T\_1], 
                                \(\(Derivative[m_, n_]\)[w_]\)[x_, t] -> 
                                  \(dt[n]\)[D[w[x, T\_0, T\_1], {x, m}]], 
                                t -> T\_0}\) /. solRule\) /. betayRule // 
                        ExpandAll)\) /. intRule3\) //. intRule4 // ExpandAll)
              \) /. \[Epsilon]\^\(n_ /; n > 1\) -> 0; )\) // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({12.6290000000001345`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq732a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqEps = 
      \(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@\ 
          eq732a // Transpose; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps[\([1]\)] /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            InterpretationBox[
              RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                SubsuperscriptBox["v", "0", 
                  
                  InterpretationBox[
                    \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                    SequenceForm[ "'", "'", "'", "'"],
                    Editable->False]]}],
              SequenceForm[ "", 
                Power[ 
                  Subscript[ v, 0], 
                  SequenceForm[ "'", "'", "'", "'"]]],
              Editable->False], "+", 
            InterpretationBox[
              RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                SubsuperscriptBox["v", "0", 
                  InterpretationBox["\[Null]",
                    SequenceForm[ ],
                    Editable->False]]}],
              SequenceForm[ 
                Power[ 
                  Subscript[ D, 0], 2], 
                Power[ 
                  Subscript[ v, 0], 
                  SequenceForm[ ]]],
              Editable->False], "+", 
            RowBox[{
              RowBox[{"(", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["v", "0", 
                      
                      InterpretationBox[
                        \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'", "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ v, 0], 
                      SequenceForm[ "'", "'", "'", "'"]]],
                  Editable->False], ")"}], " ", \(\[Delta]\_0\)}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            InterpretationBox[
              RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                SubsuperscriptBox["w", "0", 
                  
                  InterpretationBox[
                    \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                    SequenceForm[ "'", "'", "'", "'"],
                    Editable->False]]}],
              SequenceForm[ "", 
                Power[ 
                  Subscript[ w, 0], 
                  SequenceForm[ "'", "'", "'", "'"]]],
              Editable->False], "+", 
            InterpretationBox[
              RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                SubsuperscriptBox["w", "0", 
                  InterpretationBox["\[Null]",
                    SequenceForm[ ],
                    Editable->False]]}],
              SequenceForm[ 
                Power[ 
                  Subscript[ D, 0], 2], 
                Power[ 
                  Subscript[ w, 0], 
                  SequenceForm[ ]]],
              Editable->False]}], "==", "0"}], ",", 
        \(v\_0[0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["v", "0", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["v", "0", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["v", "0", 
              TagBox[\((3, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", \(w\_0[0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "0", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "0", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "0", 
              TagBox[\((3, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "In the presence of damping, all modes that are not directly excited by the \
forcing or indirectly excited by the internal resonance will decay with time. \
Hence, the solution of ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be expressed in terms of the two excited linear free-vibration modes; \
that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol0 = {
        v\_0 -> Function[{x, T\_0, T\_1}, 
            A\_1[T\_1]\ \[Phi]\_m[x]\ Exp[I\ \[Omega]\_\(1, m\)\ T\_0] + 
              A\&_\_1[T\_1]\ \[Phi]\_m[x]\ 
                Exp[\(-I\)\ \[Omega]\_\(1, m\)\ T\_0]], 
        w\_0 -> Function[{x, T\_0, T\_1}, 
            A\_2[T\_1]\ \[Phi]\_n[x]\ Exp[I\ \[Omega]\_\(2, n\)\ T\_0] + 
              A\&_\_2[T\_1]\ \[Phi]\_n[x]\ 
                Exp[\(-I\)\ \[Omega]\_\(2, n\)\ T\_0]]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "One can easily show that the mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_i[x]\)]],
  " are orthogonal. We assume that these modes are normalized so that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[Cell[BoxData[
    \(\[Integral]\_0\%1 \[Phi]\_i[x]\ \[Phi]\_j[x]\ \[DifferentialD]x -> 
      \[Delta]\_ij\)], "Input"]], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\_ij\)]],
  " is the ",
  "Kronecker",
  " delta function."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol0",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the first-order equations, ",
  StyleBox["eqEps[[2,1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eqEps[[2,2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and using ",
  StyleBox["intRule3",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule4",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Clear[order1Eq]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(order1Eq[1] = 
      \((eqEps[\([1, 1, 1]\)] /. v\_0 -> v\_1)\) == 
        \((\(\((eqEps[\([1, 1, 1]\)] /. v\_0 -> v\_1)\) - 
                  \((\(Subtract\ @@\ eqEps[\([2, 1]\)] /. sol0 // TrigToExp
                        \) // Expand)\) /. intRule3\) //. intRule4 // Expand)
          \); \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(order1Eq[2] = 
      \((eqEps[\([1, 2, 1]\)] /. w\_0 -> w\_1)\) == 
        \((\(\((eqEps[\([1, 2, 1]\)] /. w\_0 -> w\_1)\) - 
                  \((\(Subtract\ @@\ eqEps[\([2, 2]\)] /. sol0 // TrigToExp
                        \) // Expand)\) /. intRule3\) //. intRule4 // Expand)
          \); \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "It follows from ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the first-order boundary conditions are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1BC = Drop[eqEps[\([2]\)], 2]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(v\_1[0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["v", "1", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["v", "1", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["v", "1", 
              TagBox[\((3, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", \(w\_1[0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((3, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1\), "]"}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We consider the case of primary resonance of the flexural mode in the ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " direction, ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[CapitalOmega] \[TildeTilde] \[Omega]\_\(1, \ m\)\)]],
  ". To describe the nearness of the primary resonance, we introduce the \
detuning parameter ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\)]],
  " by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(omgList = {\[Omega]\_\(1, m\), \[Omega]\_\(2, n\)}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {\[Omega]\_\(2, n\) == \[Omega]\_\(1, m\), 
        \[CapitalOmega] == \[Omega]\_\(1, m\) + \[Epsilon]\ \[Sigma]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and define the following rules:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_\(2, n\) \[Rule] \[Omega]\_\(1, m\), 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma] + \[Omega]\_\(1, m\)}, {
        \[Omega]\_\(1, m\) \[Rule] \[Omega]\_\(2, n\), 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma] + \[Omega]\_\(2, n\)}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand side of ",
  StyleBox["order1Eq[1]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order1Eq[2]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the source of secular terms as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ST = 
      Table[Coefficient[\(order1Eq[i]\)[\([2]\)] /. expRule[i], 
          Exp[I\ omgList[\([i]\)]\ T\_0]], {i, 2}]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The solvability conditions of ",
  StyleBox["order1Eq[i]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order1BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demand that ",
  StyleBox["ST[[i]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_i[x]\)]],
  ". Imposing these conditions, we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SCond1 = 
      \(int[\[Phi]\_m[x]\ ST[\([1]\)], {x, 0, 1}] == 0 /. intRule1\) //. 
        intRule2; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SCond2 = 
      \(int[\[Phi]\_n[x]\ ST[\([2]\)], {x, 0, 1}] == 0 /. intRule1\) //. 
        intRule2; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Recall that the ",
  Cell[BoxData[
      \(TraditionalForm\`i\)]],
  "th spatial derivative should be recovered back from ",
  Cell[BoxData[
      \(TraditionalForm\`int[i]\)]],
  " as stated in the preceding section. Hence, we define"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(intRule5[mode_] = 
      int[mode\ a_\ \(int[i_]\)[b_, c__], d_] :> 
        int[mode\ HoldForm[D[a\ int[b, c], {x, i}]], d]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and rewrite ",
  StyleBox["SCond1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["SCond2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SCond = {SCond1 /. intRule5[\[Phi]\_m[x]], 
        SCond2 /. intRule5[\[Phi]\_n[x]]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To simplify the notation in the solvability conditions, we use the \
orthonormality of the mode shapes and define the following parameters:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"notationRule", "=", 
        RowBox[{"{", 
          RowBox[{\(int[\[Phi]\_i_[x]\^2, {x, 0, 1}] -> 1\), ",", 
            RowBox[{
              RowBox[{"int", "[", 
                RowBox[{
                  RowBox[{\(\[Phi]\_i_[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "i_", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], ",", 
                  \({x, 0, 1}\)}], "]"}], "->", \(\[Lambda]\_i\%4\)}], ",", 
            \(int[\[Mu][x]\ \[Phi]\_i_[x]\^2, {x, 0, 1}] -> \[Mu]\_i\), ",", 
            \(int[F[x]\ \[Phi]\_i_[x], {x, 0, 1}] -> 
              \[Omega]\_\(1, i\)\ f\)}], "}"}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To identify the forms of the nonlinear terms in the solvability conditions, \
we first identify all of the possible forms of the nonhomogeneous terms in \
the first-order problem. To this end, we let\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(basicTerms = 
      Table[{A\_i[T\_1]\ E\^\(I\ omgList[\([i]\)]\ T\_0\), 
            A\&_\_i[T\_1]\ E\^\(\(-I\)\ omgList[\([i]\)]\ T\_0\)}, {i, 2}] // 
        Flatten\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(I\ T\_0\ \[Omega]\_\(1, m\)\)\ A\_1[T\_1], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_\(1, m\)\)\ A\&_\_1[T\_1], 
      E\^\(I\ T\_0\ \[Omega]\_\(2, n\)\)\ A\_2[T\_1], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_\(2, n\)\)\ A\&_\_2[T\_1]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Then, all of the possible forms of the nonlinear terms in the first-order \
problem are given by\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cubicTerms = 
      \(Nest[Outer[Times, basicTerms, #]\ &, basicTerms, 2] // Flatten\) // 
        Union; \ncubicTerms /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(3\ I\ T\_0\ \[Omega]\_\(1, m\)\)\ A\_1\%3, 
      E\^\(I\ T\_0\ \((2\ \[Omega]\_\(1, m\) + \[Omega]\_\(2, n\))\)\)\ 
        A\_1\%2\ A\_2, 
      E\^\(I\ T\_0\ \((\[Omega]\_\(1, m\) + 2\ \[Omega]\_\(2, n\))\)\)\ A\_1\ 
        A\_2\%2, E\^\(3\ I\ T\_0\ \[Omega]\_\(2, n\)\)\ A\_2\%3, 
      E\^\(I\ T\_0\ \[Omega]\_\(1, m\)\)\ A\_1\%2\ A\&_\_1, 
      E\^\(I\ T\_0\ \[Omega]\_\(2, n\)\)\ A\_1\ A\_2\ A\&_\_1, 
      E\^\(I\ T\_0\ \((\(-\[Omega]\_\(1, m\)\) + 2\ \[Omega]\_\(2, n\))\)\)\ 
        A\_2\%2\ A\&_\_1, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_\(1, m\)\)\ A\_1\ A\&_\_1\%2, 
      E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_\(1, m\) + \[Omega]\_\(2, n\))\)\)\ 
        A\_2\ A\&_\_1\%2, 
      E\^\(\(-3\)\ I\ T\_0\ \[Omega]\_\(1, m\)\)\ A\&_\_1\%3, 
      E\^\(I\ T\_0\ \((2\ \[Omega]\_\(1, m\) - \[Omega]\_\(2, n\))\)\)\ 
        A\_1\%2\ A\&_\_2, 
      E\^\(I\ T\_0\ \[Omega]\_\(1, m\)\)\ A\_1\ A\_2\ A\&_\_2, 
      E\^\(I\ T\_0\ \[Omega]\_\(2, n\)\)\ A\_2\%2\ A\&_\_2, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_\(2, n\)\)\ A\_1\ A\&_\_1\ A\&_\_2, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_\(1, m\)\)\ A\_2\ A\&_\_1\ A\&_\_2, 
      E\^\(I\ T\_0\ \((\(-2\)\ \[Omega]\_\(1, m\) - \[Omega]\_\(2, n\))\)\)\ 
        A\&_\_1\%2\ A\&_\_2, 
      E\^\(I\ T\_0\ \((\[Omega]\_\(1, m\) - 2\ \[Omega]\_\(2, n\))\)\)\ A\_1\ 
        A\&_\_2\%2, 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_\(2, n\)\)\ A\_2\ A\&_\_2\%2, 
      E\^\(I\ T\_0\ \((\(-\[Omega]\_\(1, m\)\) - 2\ \[Omega]\_\(2, n\))\)\)\ 
        A\&_\_1\ A\&_\_2\%2, 
      E\^\(\(-3\)\ I\ T\_0\ \[Omega]\_\(2, n\)\)\ A\&_\_2\%3}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Out of these terms, only the terms that may lead to secular terms appear in \
the solvability conditions, which can be identified according to\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(secularTerms = 
      \(\((\(\(\(E\^\(\(-I\)\ omgList[\([#]\)]\ T\_0\)\) cubicTerms /. 
                    expRule[#]\) /. Exp[_\ T\_0 + _. ] -> 0 // Union\) // 
              Rest)\)\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{A\_1[T\_1]\^2\ A\&_\_1[T\_1], A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1], 
        A\_2[T\_1]\^2\ A\&_\_1[T\_1], A\_1[T\_1]\^2\ A\&_\_2[T\_1], 
        A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1], 
        A\_2[T\_1]\^2\ A\&_\_2[T\_1]}, {A\_1[T\_1]\^2\ A\&_\_1[T\_1], 
        A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1], A\_2[T\_1]\^2\ A\&_\_1[T\_1], 
        A\_1[T\_1]\^2\ A\&_\_2[T\_1], A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1], 
        A\_2[T\_1]\^2\ A\&_\_2[T\_1]}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Next, we define the following parameters:", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(coef = 
      Table[Coefficient[SCond[\([i, 1]\)], secularTerms[\([i]\)]], {i, 2}] // 
        Collect[#, \[Delta]\_0]\ &; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Clear[symbolList]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolList[1] = 
      MapIndexed[
        If[#1 =!= 0, 8  \[Omega]\_\(1, m\)\ \[Alpha]\_\(1, #2[\([1]\)]\), 0]
            \ &, coef[\([1]\)]]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({8\ \[Alpha]\_\(1, 1\)\ \[Omega]\_\(1, m\), 0, 
      8\ \[Alpha]\_\(1, 3\)\ \[Omega]\_\(1, m\), 0, 
      8\ \[Alpha]\_\(1, 5\)\ \[Omega]\_\(1, m\), 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolList[2] = 
      MapIndexed[
        If[#1 =!= 0, 8  \[Omega]\_\(2, n\)\ \[Alpha]\_\(2, #2[\([1]\)]\), 0]
            \ &, coef[\([2]\)]]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({0, 8\ \[Alpha]\_\(2, 2\)\ \[Omega]\_\(2, n\), 0, 
      8\ \[Alpha]\_\(2, 4\)\ \[Omega]\_\(2, n\), 0, 
      8\ \[Alpha]\_\(2, 6\)\ \[Omega]\_\(2, n\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "and express ",
  StyleBox["SCond",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqMod = 
      \(Expand[\(-\(1\/omgList[\([#]\)]\)\) 
                \((\((\(SCond[\([#, 1]\)] /. notationRule\) /. 
                        Thread[secularTerms[\([#]\)] -> 0])\) + 
                    symbolList[#] . secularTerms[\([#]\)])\)] == 0\ &\)\ /@
        \ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(1\/2\ E\^\(I\ T\_1\ \[Sigma]\)\ f\), "+", 
            \(2\ I\ \[Mu]\_m\ A\_1[T\_1]\), "+", 
            \(\(\[Delta]\_1\ \[Lambda]\_m\%4\ A\_1[T\_1]\)\/\[Omega]\_\(1, m
                  \)\), "-", 
            \(8\ \[Alpha]\_\(1, 1\)\ A\_1[T\_1]\^2\ A\&_\_1[T\_1]\), "-", 
            \(8\ \[Alpha]\_\(1, 3\)\ A\_2[T\_1]\^2\ A\&_\_1[T\_1]\), "-", 
            \(8\ \[Alpha]\_\(1, 5\)\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1]\), 
            "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(2\ I\ \[Mu]\_n\ A\_2[T\_1]\), "-", 
            \(8\ \[Alpha]\_\(2, 2\)\ A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1]\), 
            "-", \(8\ \[Alpha]\_\(2, 4\)\ A\_1[T\_1]\^2\ A\&_\_2[T\_1]\), 
            "-", \(8\ \[Alpha]\_\(2, 6\)\ A\_2[T\_1]\^2\ A\&_\_2[T\_1]\), 
            "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{
                SubsuperscriptBox["A", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((\(\(symbolList[1] -> coef[\([1]\)] // Thread\) // Union\) // Rest)
          \) /. int -> Integrate // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(29.6829999999999927`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
            \(8\ \[Alpha]\_\(1, 1\)\ \[Omega]\_\(1, m\)\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-3\), " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "3"]}], \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"12", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"3", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    RowBox[{
                      RowBox[{\(-3\), " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              SuperscriptBox[
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], "3"]}], \(\[DifferentialD]x\)}]}]}], 
                      "-", 
                      RowBox[{"12", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", 
                                  TagBox[\((3)\),
                                    Derivative],
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}]}], "-", 
                      RowBox[{"3", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              SuperscriptBox[
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], "2"], " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", 
                                  TagBox[\((4)\),
                                    Derivative],
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}]}]}], 
                    ")"}], " ", \(\[Delta]\_0\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[Prime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  \(\[Omega]\_\(1, m\)\%2\)}]}]}], ",", 
            RowBox[{
            \(8\ \[Alpha]\_\(1, 3\)\ \[Omega]\_\(1, m\)\), "\[Rule]", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "n", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"]}], \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_m[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", 
                          TagBox[\((3)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], "-", 
                RowBox[{"3", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_m[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", 
                          TagBox[\((4)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    RowBox[{
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 1}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      SubsuperscriptBox["\[Phi]", "n", 
                                        TagBox[\((3)\),
                                        Derivative],
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                          \(\[DifferentialD]x\)}]}], "-", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 1}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      
                                      SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_1\%x\), 
                                      RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                          \(\[DifferentialD]x\)}]}], "-", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{\(\[Phi]\_m[x]\), " ", 
                            RowBox[{
                              
                              SubsuperscriptBox["\[Phi]", "m", 
                                "\[DoublePrime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            SuperscriptBox[
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "n", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              "2"]}], \(\[DifferentialD]x\)}]}], "-", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{\(\[Phi]\_m[x]\), " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              
                              SubsuperscriptBox["\[Phi]", "m", 
                                "\[DoublePrime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", 
                                TagBox[\((3)\),
                                  Derivative],
                                MultilineFunction->None], "[", "x", "]"}]}], 
                          \(\[DifferentialD]x\)}]}], "-", 
                      RowBox[{"3", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "n", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", 
                                  TagBox[\((3)\),
                                    Derivative],
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}]}], "-", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{\(\[Phi]\_m[x]\), " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", 
                                TagBox[\((4)\),
                                  Derivative],
                                MultilineFunction->None], "[", "x", "]"}]}], 
                          \(\[DifferentialD]x\)}]}]}], ")"}], " ", 
                  \(\[Delta]\_0\)}], "-", 
                FractionBox[
                  RowBox[{
                    RowBox[{"(", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 2}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      
                                      SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                          \(\[DifferentialD]x\)}]}], ")"}], " ", 
                    \(\[Delta]\_0\%2\)}], \(\[Beta]\_\[Gamma]\)], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[Prime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  \(\[Omega]\_\(2, n\)\%2\)}]}]}], ",", 
            RowBox[{
            \(8\ \[Alpha]\_\(1, 5\)\ \[Omega]\_\(1, m\)\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "n", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"]}], \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"6", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    RowBox[{
                      RowBox[{"2", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{
                              TagBox[
                                RowBox[{\(\[PartialD]\_{x, 1}\), 
                                  RowBox[{"(", 
                                    RowBox[{
                                      RowBox[{
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        TagBox[\((3)\),
                                        Derivative],
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                      RowBox[{\(\[Integral]\_0\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                    ")"}]}],
                                HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                            \(\[DifferentialD]x\)}]}]}], "-", 
                      RowBox[{"2", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{
                              TagBox[
                                RowBox[{\(\[PartialD]\_{x, 1}\), 
                                  RowBox[{"(", 
                                    RowBox[{
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                    ")"}]}],
                                HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                            \(\[DifferentialD]x\)}]}]}], "-", 
                      RowBox[{"2", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              SuperscriptBox[
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], "2"]}], \(\[DifferentialD]x\)}]}]}], 
                      "-", 
                      RowBox[{"2", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", 
                                  TagBox[\((3)\),
                                    Derivative],
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}]}], "-", 
                      RowBox[{"6", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "n", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", 
                                  TagBox[\((3)\),
                                    Derivative],
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}]}], "-", 
                      RowBox[{"2", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{\(\[Phi]\_m[x]\), " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", 
                                  TagBox[\((4)\),
                                    Derivative],
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}]}]}], 
                    ")"}], " ", \(\[Delta]\_0\)}], "-", 
                FractionBox[
                  RowBox[{"2", " ", 
                    RowBox[{"(", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 2}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      
                                      SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                          \(\[DifferentialD]x\)}]}], ")"}], " ", 
                    \(\[Delta]\_0\%2\)}], \(\[Beta]\_\[Gamma]\)]}]}]}], 
          "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((\(\(symbolList[2] -> coef[\([2]\)] // Thread\) // Union\) // Rest)
          \) /. int -> Integrate // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(22.2820000000001527`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
            \(8\ \[Alpha]\_\(2, 2\)\ \[Omega]\_\(2, n\)\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-2\), " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"6", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    RowBox[{
                      RowBox[{\(-2\), " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{
                              TagBox[
                                RowBox[{\(\[PartialD]\_{x, 1}\), 
                                  RowBox[{"(", 
                                    RowBox[{
                                      RowBox[{
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        TagBox[\((3)\),
                                        Derivative],
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                      RowBox[{\(\[Integral]\_0\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                    ")"}]}],
                                HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                            \(\[DifferentialD]x\)}]}]}], "+", 
                      RowBox[{"2", " ", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{
                              TagBox[
                                RowBox[{\(\[PartialD]\_{x, 1}\), 
                                  RowBox[{"(", 
                                    RowBox[{
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                    ")"}]}],
                                HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                            \(\[DifferentialD]x\)}]}]}]}], ")"}], " ", 
                  \(\[Delta]\_0\)}], "-", 
                FractionBox[
                  RowBox[{"2", " ", 
                    RowBox[{"(", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 2}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      
                                      SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                          \(\[DifferentialD]x\)}]}], ")"}], " ", 
                    \(\[Delta]\_0\%2\)}], \(\[Beta]\_\[Gamma]\)]}]}], ",", 
            RowBox[{
            \(8\ \[Alpha]\_\(2, 4\)\ \[Omega]\_\(2, n\)\), "\[Rule]", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"3", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_n[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", 
                          TagBox[\((3)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], "-", 
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_n[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", 
                          TagBox[\((4)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    RowBox[{
                      RowBox[{"-", 
                        RowBox[{\(\[Integral]\_0\%1\), 
                          RowBox[{
                            RowBox[{
                              TagBox[
                                RowBox[{\(\[PartialD]\_{x, 1}\), 
                                  RowBox[{"(", 
                                    RowBox[{
                                      RowBox[{
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        TagBox[\((3)\),
                                        Derivative],
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                      RowBox[{\(\[Integral]\_0\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                    ")"}]}],
                                HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                            \(\[DifferentialD]x\)}]}]}], "+", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 1}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      
                                      SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_1\%x\), 
                                      RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                          \(\[DifferentialD]x\)}]}]}], ")"}], " ", 
                  \(\[Delta]\_0\)}], "-", 
                FractionBox[
                  RowBox[{
                    RowBox[{"(", 
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 2}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      
                                      SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                  ")"}]}],
                              HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                          \(\[DifferentialD]x\)}]}], ")"}], " ", 
                    \(\[Delta]\_0\%2\)}], \(\[Beta]\_\[Gamma]\)], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[Prime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  \(\[Omega]\_\(1, m\)\%2\)}]}]}], ",", 
            RowBox[{
            \(8\ \[Alpha]\_\(2, 6\)\ \[Omega]\_\(2, n\)\), "\[Rule]", 
              RowBox[{
                RowBox[{\(-3\), " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "n", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "3"]}], \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"12", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "-", 
                RowBox[{"3", " ", 
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[Prime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}]}], 
                                ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}], ")"}], " ", 
                  \(\[Omega]\_\(2, n\)\%2\)}]}]}]}], "}"}]}], "}"}]], "Output",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox["Modulation Equations in Polar Form",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell[TextData[{
  "Using ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and the function ",
  StyleBox["PolarForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " defined in the preceding section, we obtain the modulation equations in \
polar form and the definitions for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_i\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(PolarForm[eqMod]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["a", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(1\/2\ \((
                  \(-f\)\ Sin[\[Gamma]\_1[T\_1]] - 2\ \[Mu]\_m\ a\_1[T\_1] + 
                    2\ Sin[\[Gamma]\_2[T\_1]]\ \[Alpha]\_\(1, 3\)\ 
                      a\_1[T\_1]\ a\_2[T\_1]\^2)\)\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["a", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_n\)\ a\_2[T\_1] - 
                Sin[\[Gamma]\_2[T\_1]]\ \[Alpha]\_\(2, 4\)\ a\_1[T\_1]\^2\ 
                  a\_2[T\_1]\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Gamma]", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\(\(\[Delta]\_1\ \[Lambda]\_m\%4\)\/\[Omega]\_\(1, m
                        \)\)\) - \(f\ Cos[\[Gamma]\_1[T\_1]]\)\/a\_1[T\_1] + 
                2\ \[Alpha]\_\(1, 1\)\ a\_1[T\_1]\^2 - 
                2\ \[Alpha]\_\(2, 2\)\ a\_1[T\_1]\^2 - 
                2\ Cos[\[Gamma]\_2[T\_1]]\ \[Alpha]\_\(2, 4\)\ 
                  a\_1[T\_1]\^2 + 
                2\ Cos[\[Gamma]\_2[T\_1]]\ \[Alpha]\_\(1, 3\)\ 
                  a\_2[T\_1]\^2 + 2\ \[Alpha]\_\(1, 5\)\ a\_2[T\_1]\^2 - 
                2\ \[Alpha]\_\(2, 6\)\ a\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Gamma]", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Sigma] - 
                \(\[Delta]\_1\ \[Lambda]\_m\%4\)\/\(2\ \[Omega]\_\(1, m\)\) - 
                \(f\ Cos[\[Gamma]\_1[T\_1]]\)\/\(2\ a\_1[T\_1]\) + 
                \[Alpha]\_\(1, 1\)\ a\_1[T\_1]\^2 + 
                Cos[\[Gamma]\_2[T\_1]]\ \[Alpha]\_\(1, 3\)\ a\_2[T\_1]\^2 + 
                \[Alpha]\_\(1, 5\)\ a\_2[T\_1]\^2\)}]}], "}"}], ",", 
        \({\[Gamma]\_1[T\_1] == T\_1\ \[Sigma] - \[Beta]\_1[T\_1], 
          \[Gamma]\_2[T\_1] == \(-2\)\ \[Beta]\_1[T\_1] + 2\ \[Beta]\_2[T\_1]}
          \)}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[StyleBox[
"Modulation Equations in Cartesian Form and Symmetry Property",
  FontWeight->"Bold",
  Background->None]], "Text",
  CellFrame->True,
  TextAlignment->Left,
  TextJustification->1,
  Background->RGBColor[0, 1, 1]],

Cell[TextData[{
  "Using ",
  StyleBox["eqMod",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and the function ",
  StyleBox["CartesianForm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " defined in the preceding section, we obtain the modulation equations in \
Cartesian form, the definitions for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Nu]\_i\)]],
  ", and the symmetry property as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(CartesianForm[eqMod]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubsuperscriptBox["p", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_m\)\ p\_1[T\_1] - \[Nu]\_1\ q\_1[T\_1] + 
                \(\[Delta]\_1\ \[Lambda]\_m\%4\ q\_1[T\_1]\)\/\(2\ 
                    \[Omega]\_\(1, m\)\) - 
                \[Alpha]\_\(1, 1\)\ p\_1[T\_1]\^2\ q\_1[T\_1] + 
                \[Alpha]\_\(1, 3\)\ p\_2[T\_1]\^2\ q\_1[T\_1] - 
                \[Alpha]\_\(1, 5\)\ p\_2[T\_1]\^2\ q\_1[T\_1] - 
                \[Alpha]\_\(1, 1\)\ q\_1[T\_1]\^3 - 
                2\ \[Alpha]\_\(1, 3\)\ p\_1[T\_1]\ p\_2[T\_1]\ q\_2[T\_1] - 
                \[Alpha]\_\(1, 3\)\ q\_1[T\_1]\ q\_2[T\_1]\^2 - 
                \[Alpha]\_\(1, 5\)\ q\_1[T\_1]\ q\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["q", "1", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\(f\/2\)\) + \[Nu]\_1\ p\_1[T\_1] - 
                \(\[Delta]\_1\ \[Lambda]\_m\%4\ p\_1[T\_1]\)\/\(2\ 
                    \[Omega]\_\(1, m\)\) + 
                \[Alpha]\_\(1, 1\)\ p\_1[T\_1]\^3 + 
                \[Alpha]\_\(1, 3\)\ p\_1[T\_1]\ p\_2[T\_1]\^2 + 
                \[Alpha]\_\(1, 5\)\ p\_1[T\_1]\ p\_2[T\_1]\^2 - 
                \[Mu]\_m\ q\_1[T\_1] + 
                \[Alpha]\_\(1, 1\)\ p\_1[T\_1]\ q\_1[T\_1]\^2 + 
                2\ \[Alpha]\_\(1, 3\)\ p\_2[T\_1]\ q\_1[T\_1]\ q\_2[T\_1] - 
                \[Alpha]\_\(1, 3\)\ p\_1[T\_1]\ q\_2[T\_1]\^2 + 
                \[Alpha]\_\(1, 5\)\ p\_1[T\_1]\ q\_2[T\_1]\^2\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["p", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\(-\[Mu]\_n\)\ p\_2[T\_1] - 
                2\ \[Alpha]\_\(2, 4\)\ p\_1[T\_1]\ p\_2[T\_1]\ q\_1[T\_1] - 
                \[Nu]\_2\ q\_2[T\_1] - 
                \[Alpha]\_\(2, 2\)\ p\_1[T\_1]\^2\ q\_2[T\_1] + 
                \[Alpha]\_\(2, 4\)\ p\_1[T\_1]\^2\ q\_2[T\_1] - 
                \[Alpha]\_\(2, 6\)\ p\_2[T\_1]\^2\ q\_2[T\_1] - 
                \[Alpha]\_\(2, 2\)\ q\_1[T\_1]\^2\ q\_2[T\_1] - 
                \[Alpha]\_\(2, 4\)\ q\_1[T\_1]\^2\ q\_2[T\_1] - 
                \[Alpha]\_\(2, 6\)\ q\_2[T\_1]\^3\)}], ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["q", "2", "\[Prime]",
                  MultilineFunction->None], "[", \(T\_1\), "]"}], "==", 
              \(\[Nu]\_2\ p\_2[T\_1] + 
                \[Alpha]\_\(2, 2\)\ p\_1[T\_1]\^2\ p\_2[T\_1] + 
                \[Alpha]\_\(2, 4\)\ p\_1[T\_1]\^2\ p\_2[T\_1] + 
                \[Alpha]\_\(2, 6\)\ p\_2[T\_1]\^3 + 
                \[Alpha]\_\(2, 2\)\ p\_2[T\_1]\ q\_1[T\_1]\^2 - 
                \[Alpha]\_\(2, 4\)\ p\_2[T\_1]\ q\_1[T\_1]\^2 - 
                \[Mu]\_n\ q\_2[T\_1] + 
                2\ \[Alpha]\_\(2, 4\)\ p\_1[T\_1]\ q\_1[T\_1]\ q\_2[T\_1] + 
                \[Alpha]\_\(2, 6\)\ p\_2[T\_1]\ q\_2[T\_1]\^2\)}]}], "}"}], 
        ",", \({\[Nu]\_1 \[Rule] \[Sigma], \[Nu]\_2 \[Rule] \[Sigma]}\), ",", 
        \({{p\_1, q\_1, p\_2, q\_2}, {p\_1, q\_1, \(-p\_2\), \(-q\_2\)}}\)}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.3.3 Discretization of the Continuous Problem", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "As an alternative, we apply the method of multiple scales to the \
discretized system of ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To determine the discretized form, we express ",
  Cell[BoxData[
      \(TraditionalForm\`v[x, t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`w[x, t]\)]],
  " in terms of the linear mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_m[x]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_n[x]\)]],
  ", respectively, as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(discretRule = {v -> Function[{x, t}, q\_m[t]\ \[Phi]\_m[x]], 
        w -> Function[{x, t}, q\_n[t]\ \[Phi]\_n[x]]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modes = {\[Phi]\_m[x], \[Phi]\_n[x]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`q\_i[t]\)]],
  " are the generalized coordinates. We consider only ",
  Cell[BoxData[
      \(TraditionalForm\`q\_m[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`q\_n[t]\)]],
  " because in the presence of damping, all other modes that are not directly \
or indirectly excited decay with time. Substituting ",
  StyleBox["discretRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into each of ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", multiplying the results with ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_m[x]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_n[x]\)]],
  ", respectively, and integrating the outcome from ",
  Cell[BoxData[
      \(TraditionalForm\`x = 0\)]],
  " to ",
  Cell[BoxData[
      \(TraditionalForm\`x = 1\)]],
  " using ",
  StyleBox["intRule3",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["intRule4",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["intRule5",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the following discretized form of the equations describing the \
response of the beam:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq733a = 
      Table[\(\(\(\(\(\(int[modes[\([k]\)]\ # /. discretRule, {x, 0, 1}] /. 
                        intRule3\) //. intRule4\) /. intRule1\) //. intRule2
                  \) /. intRule5[modes[\([k]\)]]\) //. 
              int[a_, b__] -> HoldForm[Integrate[a, b]]\ &\)\ /@\ 
          EOM[\([k]\)], {k, 2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\)}], 
            "+", 
            RowBox[{
              RowBox[{"(", 
                
                TagBox[\(\[Integral]\_0\%1\( \[Phi]\_m[x]\^2\) 
                      \[DifferentialD]x\),
                  HoldForm], ")"}], " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}], "==", 
          RowBox[{
            RowBox[{\(-\[Epsilon]\), " ", \(Cos[t\ \[CapitalOmega]]\), " ", 
              
              TagBox[\(\[Integral]\_0\%1\( F[x]\ \[Phi]\_m[x]\) 
                    \[DifferentialD]x\),
                HoldForm]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "3"]}], \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", 
              \(q\_m[t]\^3\)}], "-", 
            RowBox[{"4", " ", "\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", 
              \(q\_m[t]\^3\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", 
              \(q\_m[t]\^3\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    TagBox[\((3)\),
                                      Derivative],
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_0\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\), " ", \(q\_n[t]\^2\)}], 
            "+", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\), " ", \(q\_n[t]\^2\)}], 
            "+", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    TagBox[\((3)\),
                                      Derivative],
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_0\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\^2\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\^2\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "n", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"]}], \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\^2\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\^2\)}], "-", 
            RowBox[{"3", " ", "\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\^2\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_m[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\^2\)}], "-", 
            FractionBox[
              RowBox[{"\[Epsilon]", " ", 
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(q\_m[t]\), " ", 
                \(q\_n[t]\^2\)}], \(\[Beta]\_\[Gamma]\)], "+", 
            FractionBox[
              RowBox[{"2", " ", "\[Epsilon]", " ", 
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\), 
                " ", \(q\_n[t]\^2\)}], \(\[Beta]\_\[Gamma]\)], "-", 
            FractionBox[
              RowBox[{"\[Epsilon]", " ", 
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(\[Beta]\_y\%2\), " ", 
                \(q\_m[t]\), " ", \(q\_n[t]\^2\)}], \(\[Beta]\_\[Gamma]\)], 
            "-", 
            RowBox[{"2", " ", "\[Epsilon]", " ", 
              RowBox[{"(", 
                
                TagBox[\(\[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_m[x]\^2\) 
                      \[DifferentialD]x\),
                  HoldForm], ")"}], " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\), " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "m", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\), " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "n", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\), " ", \(q\_n[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}]}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\)}], "+", 
            RowBox[{
              RowBox[{"(", 
                
                TagBox[\(\[Integral]\_0\%1\( \[Phi]\_n[x]\^2\) 
                      \[DifferentialD]x\),
                  HoldForm], ")"}], " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}], "==", 
          RowBox[{
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    TagBox[\((3)\),
                                      Derivative],
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_0\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], 
            "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], 
            "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], 
            "-", 
            RowBox[{"3", " ", "\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], 
            "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], 
            "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], 
            "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    TagBox[\((3)\),
                                      Derivative],
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_0\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\^2\), 
              " ", \(q\_n[t]\)}], "+", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{
                                      RowBox[{
                                        RowBox[{
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", \(q\_m[t]\^2\), 
              " ", \(q\_n[t]\)}], "-", 
            FractionBox[
              RowBox[{"\[Epsilon]", " ", 
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(q\_m[t]\^2\), " ", 
                \(q\_n[t]\)}], \(\[Beta]\_\[Gamma]\)], "+", 
            FractionBox[
              RowBox[{"2", " ", "\[Epsilon]", " ", 
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(\[Beta]\_y\), " ", 
                \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], \(\[Beta]\_\[Gamma]\)], 
            "-", 
            FractionBox[
              RowBox[{"\[Epsilon]", " ", 
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(\[Beta]\_y\%2\), " ", 
                \(q\_m[t]\^2\), " ", \(q\_n[t]\)}], \(\[Beta]\_\[Gamma]\)], 
            "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "n", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "3"]}], \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\^3\)}], "-", 
            RowBox[{"4", " ", "\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((3)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\^3\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{\(\[Phi]\_n[x]\), " ", 
                        SuperscriptBox[
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          "2"], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", 
                            TagBox[\((4)\),
                              Derivative],
                            MultilineFunction->None], "[", "x", "]"}]}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\^3\)}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\), " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "m", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "-", 
            RowBox[{"2", " ", "\[Epsilon]", " ", 
              RowBox[{"(", 
                
                TagBox[\(\[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_n[x]\^2\) 
                      \[DifferentialD]x\),
                  HoldForm], ")"}], " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\), " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "n", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_m[t]\), " ", \(q\_n[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "-", 
            RowBox[{"\[Epsilon]", " ", 
              RowBox[{"(", 
                TagBox[
                  RowBox[{\(\[Integral]\_0\%1\), 
                    RowBox[{
                      RowBox[{
                        TagBox[
                          RowBox[{\(\[PartialD]\_{x, 1}\), 
                            RowBox[{"(", 
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[Prime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                TagBox[
                                  RowBox[{\(\[Integral]\_1\%x\), 
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        SuperscriptBox[
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                  HoldForm]}], ")"}]}],
                          HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                      \(\[DifferentialD]x\)}]}],
                  HoldForm], ")"}], " ", \(q\_n[t]\^2\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the damping is assumed to be modal. Using the orthonormality of the \
mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_i[x]\)]],
  ", we define"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"notationRule", "=", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{\(int[\[Phi]\_i_[x]\^2, {x, 0, 1}] -> 1\), ",", 
            RowBox[{
              RowBox[{"int", "[", 
                RowBox[{
                  RowBox[{\(\[Phi]\_i_[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "i_", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], ",", 
                  \({x, 0, 1}\)}], "]"}], "->", \(\[Lambda]\_i\%4\)}], ",", 
            \(int[\[Mu][x]\ \[Phi]\_i_[x]\^2, {x, 0, 1}] -> \[Mu]\_i\), ",", 
            \(int[F[x]\ \[Phi]\_i_[x], {x, 0, 1}] -> 
              \[Omega]\_\(1, i\)\ f\)}], "}"}], "/.", 
        \(int[a_, b__] -> HoldForm[Integrate[a, b]]\)}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          
          TagBox[\(\[Integral]\_0\%1\( \[Phi]\_i_[x]\^2\) 
                \[DifferentialD]x\),
            HoldForm], "\[Rule]", "1"}], ",", 
        RowBox[{
          TagBox[
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{\(\[Phi]\_i_[x]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "i_", 
                      TagBox[\((4)\),
                        Derivative],
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}],
            HoldForm], "\[Rule]", \(\[Lambda]\_i\%4\)}], ",", 
        RowBox[{
          
          TagBox[\(\[Integral]\_0\%1\( \[Mu][x]\ \[Phi]\_i_[x]\^2\) 
                \[DifferentialD]x\),
            HoldForm], "\[Rule]", \(\[Mu]\_i\)}], ",", 
        RowBox[{
          
          TagBox[\(\[Integral]\_0\%1\( F[x]\ \[Phi]\_i_[x]\) 
                \[DifferentialD]x\),
            HoldForm], "\[Rule]", \(f\ \[Omega]\_\(1, i\)\)}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To simplify the expression for ",
  StyleBox["eq733a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we first find possible nonlinear terms as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(basicTerms = Table[D[q\_m[t], {t, i}], {i, 0, 2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(q\_m[t]\), ",", 
        RowBox[{
          SubsuperscriptBox["q", "m", "\[Prime]",
            MultilineFunction->None], "[", "t", "]"}], ",", 
        RowBox[{
          SubsuperscriptBox["q", "m", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(linearTerms = Join[basicTerms, basicTerms /. m -> n]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(q\_m[t]\), ",", 
        RowBox[{
          SubsuperscriptBox["q", "m", "\[Prime]",
            MultilineFunction->None], "[", "t", "]"}], ",", 
        RowBox[{
          SubsuperscriptBox["q", "m", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}], ",", \(q\_n[t]\), 
        ",", 
        RowBox[{
          SubsuperscriptBox["q", "n", "\[Prime]",
            MultilineFunction->None], "[", "t", "]"}], ",", 
        RowBox[{
          SubsuperscriptBox["q", "n", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(\(cubicTerms = 
      \(\[Epsilon]\ Nest[Outer[Times, linearTerms, #]\ &, linearTerms, 2] // 
          Flatten\) // Union; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The coefficients corresponding to ",
  StyleBox["cubicTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in ",
  StyleBox["eq733a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are obtained as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(coefList = 
      \(Coefficient[eq733a[\([#, 2]\)], cubicTerms]\ &\)\ /@\ {1, 2}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We replace these coefficients by ",
  Cell[BoxData[
      \(TraditionalForm\`\[Alpha]\_\(i, \ j\)\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(symbolList = 
      MapIndexed[If[#1 =!= 0, \[Alpha]\_\(Sequence\ @@\ #2\), 0]\ &, 
        coefList, {2}]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["notationRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["symbolList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["cubicTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we can rewrite ",
  StyleBox["eq733a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(EOM1 = 
      Table[eq733a[\([k, 1]\)] == 
            \((eq733a[\([k, 2]\)] /. Thread[cubicTerms -> 0])\) + 
              symbolList[\([k]\)] . cubicTerms /. notationRule, {k, 2}]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(\[Beta]\_y\ \[Lambda]\_m\%4\ q\_m[t]\), "+", 
            RowBox[{
              SubsuperscriptBox["q", "m", "\[DoublePrime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", 
          RowBox[{
          \(\(-f\)\ \[Epsilon]\ Cos[t\ \[CapitalOmega]]\ 
              \[Omega]\_\(1, m\)\), "+", 
            \(\[Epsilon]\ \[Alpha]\_\(1, 1\)\ q\_m[t]\^3\), "+", 
            \(\[Epsilon]\ \[Alpha]\_\(1, 3\)\ q\_m[t]\ q\_n[t]\^2\), "-", 
            RowBox[{"2", " ", "\[Epsilon]", " ", \(\[Mu]\_m\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(1, 8\)\), " ", \(q\_m[t]\), 
              " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "m", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(1, 17\)\), " ", \(q\_m[t]\), 
              " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "n", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(1, 21\)\), " ", \(q\_m[t]\^2\), 
              " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(1, 37\)\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}]}], ",", 
        RowBox[{
          RowBox[{\(\[Lambda]\_n\%4\ q\_n[t]\), "+", 
            RowBox[{
              SubsuperscriptBox["q", "n", "\[DoublePrime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", 
          RowBox[{
          \(\[Epsilon]\ \[Alpha]\_\(2, 2\)\ q\_m[t]\^2\ q\_n[t]\), "+", 
            \(\[Epsilon]\ \[Alpha]\_\(2, 4\)\ q\_n[t]\^3\), "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(2, 9\)\), " ", \(q\_n[t]\), 
              " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "m", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "-", 
            RowBox[{"2", " ", "\[Epsilon]", " ", \(\[Mu]\_n\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(2, 18\)\), " ", \(q\_n[t]\), 
              " ", 
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["q", "n", "\[Prime]",
                    MultilineFunction->None], "[", "t", "]"}], "2"]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(2, 22\)\), " ", \(q\_m[t]\), 
              " ", \(q\_n[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["q", "m", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{
            "\[Epsilon]", " ", \(\[Alpha]\_\(2, 38\)\), " ", \(q\_n[t]\^2\), 
              " ", 
              RowBox[{
                SubsuperscriptBox["q", "n", "\[DoublePrime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Using the method of multiple scales, we seek a first-order uniform expansion \
in the form\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = 
      q\_n_ -> \((\[Sum]\+\(j = 0\)\%1 \[Epsilon]\^j\ q\_\(n, j\)[#1, #2]\ &)
          \); \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Transforming the total time derivatives in ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into partial derivatives in terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", substituting the ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["betayRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than \[Epsilon], we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq733b = 
      \((\(\(EOM1 /. {q\_s_[t] -> q\_s[T\_0, T\_1], 
                    \(\(Derivative[n_]\)[q\_s_]\)[t] -> 
                      \(dt[n]\)[q\_s[T\_0, T\_1]], t -> T\_0}\) /. solRule
                \) /. betayRule // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 1\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq733b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      \(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@\ 
          eq733b // Transpose\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\[Lambda]\_m\%4\ q\_\(m, 0\)[T\_0, T\_1]\), "+", 
                \(\[Delta]\_0\ \[Lambda]\_m\%4\ q\_\(m, 0\)[T\_0, T\_1]\), 
                "+", 
                RowBox[{
                  SubsuperscriptBox["q", \(m, 0\), 
                    TagBox[\((2, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{\(\[Lambda]\_n\%4\ q\_\(n, 0\)[T\_0, T\_1]\), "+", 
                RowBox[{
                  SubsuperscriptBox["q", \(n, 0\), 
                    TagBox[\((2, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
              "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(f\ Cos[T\_0\ \[CapitalOmega]]\ \[Omega]\_\(1, m\)\), "+", 
                \(\[Delta]\_1\ \[Lambda]\_m\%4\ q\_\(m, 0\)[T\_0, T\_1]\), 
                "-", \(\[Alpha]\_\(1, 1\)\ q\_\(m, 0\)[T\_0, T\_1]\^3\), "+", 
                \(\[Lambda]\_m\%4\ q\_\(m, 1\)[T\_0, T\_1]\), "+", 
                \(\[Delta]\_0\ \[Lambda]\_m\%4\ q\_\(m, 1\)[T\_0, T\_1]\), 
                "-", \(\[Alpha]\_\(1, 3\)\ q\_\(m, 0\)[T\_0, T\_1]\ 
                  q\_\(n, 0\)[T\_0, T\_1]\^2\), "+", 
                RowBox[{"2", " ", \(\[Mu]\_m\), " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(m, 0\), 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "-", 
                RowBox[{
                \(\[Alpha]\_\(1, 8\)\), " ", \(q\_\(m, 0\)[T\_0, T\_1]\), 
                  " ", 
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["q", \(m, 0\), 
                        TagBox[\((1, 0)\),
                          Derivative],
                        MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}], 
                    "2"]}], "-", 
                RowBox[{
                \(\[Alpha]\_\(1, 17\)\), " ", \(q\_\(m, 0\)[T\_0, T\_1]\), 
                  " ", 
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["q", \(n, 0\), 
                        TagBox[\((1, 0)\),
                          Derivative],
                        MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}], 
                    "2"]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(m, 0\), 
                      TagBox[\((1, 1)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "-", 
                RowBox[{
                \(\[Alpha]\_\(1, 21\)\), " ", \(q\_\(m, 0\)[T\_0, T\_1]\^2\), 
                  " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(m, 0\), 
                      TagBox[\((2, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "+", 
                RowBox[{
                  SubsuperscriptBox["q", \(m, 1\), 
                    TagBox[\((2, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}], 
                "-", 
                RowBox[{
                \(\[Alpha]\_\(1, 37\)\), " ", \(q\_\(m, 0\)[T\_0, T\_1]\), 
                  " ", \(q\_\(n, 0\)[T\_0, T\_1]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(n, 0\), 
                      TagBox[\((2, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), 
                    "]"}]}]}], "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-\[Alpha]\_\(2, 2\)\)\ q\_\(m, 0\)[T\_0, T\_1]\^2\ 
                  q\_\(n, 0\)[T\_0, T\_1]\), "-", 
                \(\[Alpha]\_\(2, 4\)\ q\_\(n, 0\)[T\_0, T\_1]\^3\), "+", 
                \(\[Lambda]\_n\%4\ q\_\(n, 1\)[T\_0, T\_1]\), "-", 
                RowBox[{
                \(\[Alpha]\_\(2, 9\)\), " ", \(q\_\(n, 0\)[T\_0, T\_1]\), 
                  " ", 
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["q", \(m, 0\), 
                        TagBox[\((1, 0)\),
                          Derivative],
                        MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}], 
                    "2"]}], "+", 
                RowBox[{"2", " ", \(\[Mu]\_n\), " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(n, 0\), 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "-", 
                RowBox[{
                \(\[Alpha]\_\(2, 18\)\), " ", \(q\_\(n, 0\)[T\_0, T\_1]\), 
                  " ", 
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["q", \(n, 0\), 
                        TagBox[\((1, 0)\),
                          Derivative],
                        MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}], 
                    "2"]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(n, 0\), 
                      TagBox[\((1, 1)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "-", 
                RowBox[{
                \(\[Alpha]\_\(2, 22\)\), " ", \(q\_\(m, 0\)[T\_0, T\_1]\), 
                  " ", \(q\_\(n, 0\)[T\_0, T\_1]\), " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(m, 0\), 
                      TagBox[\((2, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "-", 
                RowBox[{
                \(\[Alpha]\_\(2, 38\)\), " ", \(q\_\(n, 0\)[T\_0, T\_1]\^2\), 
                  " ", 
                  RowBox[{
                    SubsuperscriptBox["q", \(n, 0\), 
                      TagBox[\((2, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
                "+", 
                RowBox[{
                  SubsuperscriptBox["q", \(n, 1\), 
                    TagBox[\((2, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_0, T\_1\), "]"}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then it follows from ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol0 = {
        q\_\(m, 0\) -> 
          Function[{T\_0, T\_1}, 
            A\_1[T\_1]\ Exp[I\ \[Omega]\_\(1, m\)\ T\_0] + 
              A\&_\_1[T\_1]\ Exp[\(-I\)\ \[Omega]\_\(1, m\)\ T\_0]], 
        q\_\(n, 0\) -> 
          Function[{T\_0, T\_1}, 
            A\_2[T\_1]\ Exp[I\ \[Omega]\_\(2, n\)\ T\_0] + 
              A\&_\_2[T\_1]\ Exp[\(-I\)\ \[Omega]\_\(2, n\)\ T\_0]]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol0",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(order1Eq = 
      \(\(\((\(#[\([1]\)]\ &\)\ /@\ eqEps[\([1]\)] /. 
                    q\_\(k_, 0\) -> q\_\(k, 1\))\) == 
                \((\(#[\([1]\)]\ &\)\ /@\ eqEps[\([1]\)] /. 
                      q\_\(k_, 0\) -> q\_\(k, 1\))\) - 
                  \((\(#[\([1]\)]\ &\)\ /@\ eqEps[\([2]\)])\) /. sol0 // 
            TrigToExp\) // ExpandAll\) // Thread; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We consider the case of primary resonance of the flexural mode in the ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " direction, ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[CapitalOmega] \[TildeTilde] \[Omega]\_\(1, \ m\)\)]],
  ". To describe the nearness of the primary resonance, we introduce the \
detuning parameter ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\)]],
  " and define"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(omgList = {\[Omega]\_\(1, m\), \[Omega]\_\(2, n\)}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds = {\[Omega]\_\(2, n\) == \[Omega]\_\(1, m\), 
        \[CapitalOmega] == \[Omega]\_\(1, m\) + \[Epsilon]\ \[Sigma]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, Drop[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
            \([1]\)]\ &\)\ /@\ {1, 2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_\(2, n\) \[Rule] \[Omega]\_\(1, m\), 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma] + \[Omega]\_\(1, m\)}, {
        \[Omega]\_\(1, m\) \[Rule] \[Omega]\_\(2, n\), 
        \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma] + \[Omega]\_\(2, n\)}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\ T\_0 -> T\_1]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We substitute ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the right-hand sides of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the solvability condition as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      Table[Coefficient[\(-order1Eq[\([i, 2]\)]\) /. expRule[i], 
            Exp[I\ omgList[\([i]\)]\ T\_0]] == 0, {i, 2}]; \n
    SCond /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(A\_1\ \[Delta]\_1\ \[Lambda]\_m\%4\), "-", 
            \(3\ A\_1\%2\ A\&_\_1\ \[Alpha]\_\(1, 1\)\), "-", 
            \(A\_2\%2\ A\&_\_1\ \[Alpha]\_\(1, 3\)\), "-", 
            \(2\ A\_1\ A\_2\ A\&_\_2\ \[Alpha]\_\(1, 3\)\), "+", 
            \(1\/2\ E\^\(I\ T\_1\ \[Sigma]\)\ f\ \[Omega]\_\(1, m\)\), "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_\(1, m\)\)}], 
            "+", \(2\ I\ A\_1\ \[Mu]\_m\ \[Omega]\_\(1, m\)\), "-", 
            \(A\_1\%2\ A\&_\_1\ \[Alpha]\_\(1, 8\)\ \[Omega]\_\(1, m\)\%2\), 
            "+", \(3\ A\_1\%2\ A\&_\_1\ \[Alpha]\_\(1, 21\)\ 
              \[Omega]\_\(1, m\)\%2\), "+", 
            \(A\_2\%2\ A\&_\_1\ \[Alpha]\_\(1, 17\)\ \[Omega]\_\(2, n\)\%2\), 
            "-", \(2\ A\_1\ A\_2\ A\&_\_2\ \[Alpha]\_\(1, 17\)\ 
              \[Omega]\_\(2, n\)\%2\), "+", 
            \(A\_2\%2\ A\&_\_1\ \[Alpha]\_\(1, 37\)\ \[Omega]\_\(2, n\)\%2\), 
            "+", \(2\ A\_1\ A\_2\ A\&_\_2\ \[Alpha]\_\(1, 37\)\ 
              \[Omega]\_\(2, n\)\%2\)}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
          \(\(-2\)\ A\_1\ A\_2\ A\&_\_1\ \[Alpha]\_\(2, 2\)\), "-", 
            \(A\_1\%2\ A\&_\_2\ \[Alpha]\_\(2, 2\)\), "-", 
            \(3\ A\_2\%2\ A\&_\_2\ \[Alpha]\_\(2, 4\)\), "-", 
            \(2\ A\_1\ A\_2\ A\&_\_1\ \[Alpha]\_\(2, 9\)\ 
              \[Omega]\_\(1, m\)\%2\), "+", 
            \(A\_1\%2\ A\&_\_2\ \[Alpha]\_\(2, 9\)\ \[Omega]\_\(1, m\)\%2\), 
            "+", \(2\ A\_1\ A\_2\ A\&_\_1\ \[Alpha]\_\(2, 22\)\ 
              \[Omega]\_\(1, m\)\%2\), "+", 
            \(A\_1\%2\ A\&_\_2\ \[Alpha]\_\(2, 22\)\ \[Omega]\_\(1, m\)\%2\), 
            "+", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_\(2, n\)\)}], 
            "+", \(2\ I\ A\_2\ \[Mu]\_n\ \[Omega]\_\(2, n\)\), "-", 
            \(A\_2\%2\ A\&_\_2\ \[Alpha]\_\(2, 18\)\ \[Omega]\_\(2, n\)\%2\), 
            "+", \(3\ A\_2\%2\ A\&_\_2\ \[Alpha]\_\(2, 38\)\ 
              \[Omega]\_\(2, n\)\%2\)}], "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(# -> 
          \((\(\(\(# /. \[Alpha]\_\(i_, j_\) :> coefList[\([i, j]\)]\) /. 
                    betayRule\) /. \[Epsilon] -> 0 // Expand\) // 
              Collect[#, \[Delta]\_0]\ &)\)\ &\)\ /@\ 
      \((\(\(symbolList // Flatten\) // Union\) // Rest)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{\(\[Alpha]\_\(1, 1\)\), "\[Rule]", 
          RowBox[{
            RowBox[{"-", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_m[x]\), " ", 
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], 
                        "3"]}], \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            RowBox[{"4", " ", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_m[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", 
                          TagBox[\((3)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_m[x]\), " ", 
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], "+", 
            RowBox[{
              RowBox[{"(", 
                RowBox[{
                  RowBox[{"-", 
                    TagBox[
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{\(\[Phi]\_m[x]\), " ", 
                            SuperscriptBox[
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              "3"]}], \(\[DifferentialD]x\)}]}],
                      HoldForm]}], "-", 
                  RowBox[{"4", " ", 
                    TagBox[
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{\(\[Phi]\_m[x]\), " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              
                              SubsuperscriptBox["\[Phi]", "m", 
                                "\[DoublePrime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", 
                                TagBox[\((3)\),
                                  Derivative],
                                MultilineFunction->None], "[", "x", "]"}]}], 
                          \(\[DifferentialD]x\)}]}],
                      HoldForm]}], "-", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{\(\[Phi]\_m[x]\), " ", 
                          SuperscriptBox[
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            "2"], " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "m", 
                              TagBox[\((4)\),
                                Derivative],
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm]}], ")"}], " ", \(\[Delta]\_0\)}]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(1, 3\)\), "\[Rule]", 
          RowBox[{
            RowBox[{"-", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_m[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], 
                        "2"]}], \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_m[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", 
                        TagBox[\((3)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], "-", 
            RowBox[{"3", " ", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_m[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", 
                          TagBox[\((3)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_m[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], "+", 
            RowBox[{
              RowBox[{"(", 
                RowBox[{
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      TagBox[\((3)\),
                                        Derivative],
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], "-", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_1\%x\), 
                                      RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], "-", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{\(\[Phi]\_m[x]\), " ", 
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          SuperscriptBox[
                            RowBox[{
                              
                              SubsuperscriptBox["\[Phi]", "n", 
                                "\[DoublePrime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            "2"]}], \(\[DifferentialD]x\)}]}],
                    HoldForm], "-", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{\(\[Phi]\_m[x]\), " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            
                            SubsuperscriptBox["\[Phi]", "m", 
                              "\[DoublePrime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "n", 
                              TagBox[\((3)\),
                                Derivative],
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], "-", 
                  RowBox[{"3", " ", 
                    TagBox[
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{\(\[Phi]\_m[x]\), " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              
                              SubsuperscriptBox["\[Phi]", "n", 
                                "\[DoublePrime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", 
                                TagBox[\((3)\),
                                  Derivative],
                                MultilineFunction->None], "[", "x", "]"}]}], 
                          \(\[DifferentialD]x\)}]}],
                      HoldForm]}], "-", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{\(\[Phi]\_m[x]\), " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                              MultilineFunction->None], "[", "x", "]"}], 
                          " ", 
                          RowBox[{
                            SubsuperscriptBox["\[Phi]", "n", 
                              TagBox[\((4)\),
                                Derivative],
                              MultilineFunction->None], "[", "x", "]"}]}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm]}], ")"}], " ", \(\[Delta]\_0\)}], "-", 
            FractionBox[
              RowBox[{
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(\[Delta]\_0\%2\)}], 
              \(\[Beta]\_\[Gamma]\)]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(1, 8\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(1, 17\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(1, 21\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(1, 37\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_m[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(2, 2\)\), "\[Rule]", 
          RowBox[{
            RowBox[{"-", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_n[x]\), " ", 
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], "2"], 
                      " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            RowBox[{"3", " ", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_n[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", 
                          TagBox[\((3)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_n[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", 
                        TagBox[\((3)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], "-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_n[x]\), " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], "+", 
            RowBox[{
              RowBox[{"(", 
                RowBox[{
                  RowBox[{"-", 
                    TagBox[
                      RowBox[{\(\[Integral]\_0\%1\), 
                        RowBox[{
                          RowBox[{
                            TagBox[
                              RowBox[{\(\[PartialD]\_{x, 1}\), 
                                RowBox[{"(", 
                                  RowBox[{
                                    RowBox[{
                                      SubsuperscriptBox["\[Phi]", "m", 
                                        TagBox[\((3)\),
                                        Derivative],
                                        MultilineFunction->None], "[", "x", 
                                      "]"}], " ", 
                                    TagBox[
                                      RowBox[{\(\[Integral]\_0\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                      HoldForm]}], ")"}]}],
                              HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                          \(\[DifferentialD]x\)}]}],
                      HoldForm]}], "+", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 1}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_1\%x\), 
                                      RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\)}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm]}], ")"}], " ", \(\[Delta]\_0\)}], "-", 
            FractionBox[
              RowBox[{
                RowBox[{"(", 
                  TagBox[
                    RowBox[{\(\[Integral]\_0\%1\), 
                      RowBox[{
                        RowBox[{
                          TagBox[
                            RowBox[{\(\[PartialD]\_{x, 2}\), 
                              RowBox[{"(", 
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  TagBox[
                                    RowBox[{\(\[Integral]\_0\%x\), 
                                      RowBox[{\(\[Integral]\_1\%x\), 
                                        RowBox[{
                                        RowBox[{
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], " ", 
                                        RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[DoublePrime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}]}], \(\[DifferentialD]x\), 
                                        \(\[DifferentialD]x\)}]}]}],
                                    HoldForm]}], ")"}]}],
                            HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                        \(\[DifferentialD]x\)}]}],
                    HoldForm], ")"}], " ", \(\[Delta]\_0\%2\)}], 
              \(\[Beta]\_\[Gamma]\)]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(2, 4\)\), "\[Rule]", 
          RowBox[{
            RowBox[{"-", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_n[x]\), " ", 
                      SuperscriptBox[
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                            MultilineFunction->None], "[", "x", "]"}], 
                        "3"]}], \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            RowBox[{"4", " ", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{\(\[Phi]\_n[x]\), " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", 
                          TagBox[\((3)\),
                            Derivative],
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm]}], "-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{\(\[Phi]\_n[x]\), " ", 
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", 
                        TagBox[\((4)\),
                          Derivative],
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(2, 9\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(2, 18\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(2, 22\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "m", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}], ",", 
        RowBox[{\(\[Alpha]\_\(2, 38\)\), "\[Rule]", 
          RowBox[{"-", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    TagBox[
                      RowBox[{\(\[PartialD]\_{x, 1}\), 
                        RowBox[{"(", 
                          RowBox[{
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            " ", 
                            TagBox[
                              RowBox[{\(\[Integral]\_1\%x\), 
                                RowBox[{\(\[Integral]\_0\%x\), 
                                  RowBox[{
                                    SuperscriptBox[
                                      RowBox[{
                                        
                                        SubsuperscriptBox["\[Phi]", "n", 
                                        "\[Prime]",
                                        MultilineFunction->None], "[", "x", 
                                        "]"}], "2"], \(\[DifferentialD]x\), 
                                    \(\[DifferentialD]x\)}]}]}],
                              HoldForm]}], ")"}]}],
                      HoldForm], " ", \(\[Phi]\_n[x]\)}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm]}]}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which is in agreement with that obtained by direct approach.", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["7.3.4 Method of Time-Averaged Lagrangian", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
As a second alternative, we derive the modulation equations by using the \
method of time-averaged Lagrangian. The nondimensional Lagrangian of the beam \
can be expressed as (Crespo da Silva and Glynn, 1978; Arafat, Nayfeh, and \
Chin, 1998)\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(Lagr1 = 
      \(Hold[\[Integral]\_0\%1
                  \((\(1\/2\) 
                        \((\[Epsilon] 
                              \((\[PartialD]\_t
                                      \(\[Integral]\_0\%x\( 1\/2\) 
                                        \((\((\[PartialD]\_x v[x, t])\)\^2 + 
                                        \((\[PartialD]\_x w[x, t])\)\^2)\) 
                                        \[DifferentialD]x\))\)\^2 + 
                            \((\[PartialD]\_t v[x, t])\)\^2 + 
                            \((\[PartialD]\_t w[x, t])\)\^2)\) - 
                      \[Epsilon] \((1 - \[Beta]\_y)\)\ 
                        \((\[PartialD]\_\(x, x\)v[x, t]\ 
                            \[PartialD]\_\(x, x\)w[x, t]\ 
                            \(\[Integral]\_0\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                                \[PartialD]\_x w[x, t] \[DifferentialD]x\))
                          \) - \(1\/2\) 
                        \((\((\[PartialD]\_\(x, x\)w[x, t])\)\^2 + 
                            \[Epsilon] \((\[PartialD]\_x w[x, t])\)\^2\ 
                              \((\[PartialD]\_\(x, x\)w[x, t])\)\^2)\) - 
                      \(1\/2\) \[Beta]\_y\ 
                        \((\((\[PartialD]\_\(x, x\)v[x, t])\)\^2 + 
                            \[Epsilon] \((\[PartialD]\_x v[x, t])\)\^2\ 
                              \((\[PartialD]\_\(x, x\)v[x, t])\)\^2 + 
                            \[Epsilon]\ 2\ \[PartialD]\_x v[x, t]\ 
                              \[PartialD]\_\(x, x\)v[x, t]\ 
                              \[PartialD]\_x w[x, t]\ 
                              \[PartialD]\_\(x, x\)w[x, t])\) - 
                      \[Epsilon]\ 
                        \(\((1 - \[Beta]\_y)\)\^2\/\(2\ \[Beta]\_\[Gamma]\)\) 
                        \((\((\[Integral]\_1\%x\[PartialD]\_\(x, x\)v[x, t]\ 
                                    \[PartialD]\_\(x, x\)w[x, t] 
                                    \[DifferentialD]x)\)\^2 + 
                            2\ \[PartialD]\_\(x, x\)v[x, t]\ 
                              \[PartialD]\_\(x, x\)w[x, t]\ 
                              \(\[Integral]\_0\%x
                                  \(\[Integral]\_1\%x
                                    \[PartialD]\_\(x, x\)v[x, t]\ 
                                    \[PartialD]\_\(x, x\)w[x, t] 
                                    \[DifferentialD]x\ \[DifferentialD]x\)\))
                          \) - \[Epsilon]\ F[x]\ Cos[\[CapitalOmega]\ t]\ 
                        v[x, t])\) \[DifferentialD]x] /. 
            Integrate -> int\) /. 
          HoldPattern[\[PartialD]\_t\ int[a_, b_]] :> 
            int[\[PartialD]\_t a, b] // ReleaseHold; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  StyleBox["Hold",
    FontWeight->"Bold"],
  " and ",
  StyleBox["int",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are used to keep ",
  StyleBox["Integrate",
    FontWeight->"Bold"],
  " from being evaluated. Transforming the total time derivatives into \
partial derivatives in terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", we modify ",
  StyleBox["Lagr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(Lagr2 = 
      Lagr1 /. {v[x_, t] -> v[x, T\_0, T\_1], w[x_, t] -> w[x, T\_0, T\_1], 
          \(\(Derivative[m_, n_]\)[w_]\)[x_, t] -> 
            \(dt[n]\)[D[w[x, T\_0, T\_1], {x, m}]], t -> T\_0}; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell["To this end, we let", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = {
        v -> Function[{x, T\_0, T\_1}, 
            A\_1[T\_1]\ \[Phi]\_m[x]\ Exp[I\ \[Omega]\_\(1, m\)\ T\_0] + 
              A\&_\_1[T\_1]\ \[Phi]\_m[x]\ 
                Exp[\(-I\)\ \[Omega]\_\(1, m\)\ T\_0]], 
        w -> Function[{x, T\_0, T\_1}, 
            A\_2[T\_1]\ \[Phi]\_n[x]\ Exp[I\ \[Omega]\_\(2, n\)\ T\_0] + 
              A\&_\_2[T\_1]\ \[Phi]\_n[x]\ 
                Exp[\(-I\)\ \[Omega]\_\(2, n\)\ T\_0]]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and define", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(intRule6 = {
        int[c1_. \ int[a1__]^n_. , b1_] :> 
          c1\ int[\(\(int[a1]^n /. intRule1\) //. intRule2\) /. 
                  \[Epsilon] -> 0, b1] /; FreeQ[c1, First[b1]], 
        int[c1_\ int[a1__]^n_. , b1_] :> 
          int[\(\(c1\ int[a1]^n /. intRule1\) //. intRule2\) /. 
              \[Epsilon] -> 0, b1]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["betayRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["Lagr2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["intRule6",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to simplify the expansions of the integrands, and collecting the \
coefficients of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((order1Lagr = 
          \(\(\(\(\(\(\(\(\(\(\(Lagr2 /. solRule\) /. betayRule\) /. intRule1
                                \) //. intRule2\) //. intRule6\) /. intRule1
                          \) //. intRule2\) //. intRule6\) /. intRule1\) //. 
                  intRule2 // TrigToExp\) // Expand\) // 
            Coefficient[#, \[Epsilon]]\ &; )\) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({14.9510000000000236`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(ResonanceConds = {\[Omega]\_\(2, n\) == \[Omega]\_\(1, m\), 
          \[CapitalOmega] == \[Omega]\_\(1, m\) + \[Epsilon]\ \[Sigma]} /. 
        Equal -> Rule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_\(2, n\) \[Rule] \[Omega]\_\(1, m\), 
      \[CapitalOmega] \[Rule] \[Epsilon]\ \[Sigma] + \[Omega]\_\(1, m\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    RowBox[{
      RowBox[{"notationRule1", "=", 
        RowBox[{"{", 
          RowBox[{
          \(int[\[Phi]\_i_[x]\^2, {x, 0, 1}] -> 1\), ",", 
            \(int[F[x]\ \[Phi]\_i_[x], {x, 0, 1}] -> f\ \[Omega]\_\(1, i\)\), 
            ",", 
            RowBox[{
              RowBox[{"int", "[", 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "i_", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
                  \({x, 0, 1}\)}], "]"}], "->", \(\[Lambda]\_i\%4\)}]}], 
          "}"}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["notationRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the slowly varying terms from ",
  StyleBox["order1Lagr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(TAL = 
      \(\(order1Lagr /. 
            Exp[a_] :> 
              Exp[Expand[a /. ResonanceConds] /. \[Epsilon]\ T\_0 -> T\_1]
                \) /. Exp[_\ T\_0 + _. ] -> 0\) /. notationRule1; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The Euler-Lagrange equations corresponding to the ",
  StyleBox["TAL",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod1", "=", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"D", "[", 
              RowBox[{
                RowBox[{"D", "[", 
                  RowBox[{"TAL", ",", 
                    RowBox[{
                      SubsuperscriptBox[\(A\&_\), "1", "\[Prime]",
                        MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                  "]"}], ",", \(T\_1\)}], "]"}], "-", 
            \(D[TAL, A\&_\_1[T\_1]]\)}], "==", "0"}], "//.", 
        \(int[a_, b__] -> HoldForm[Integrate[a, b]]\)}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(1\/2\ E\^\(I\ T\_1\ \[Sigma]\)\ f\ \[Omega]\_\(1, m\)\), "+", 
        \(\[Delta]\_1\ \[Lambda]\_m\%4\ A\_1[T\_1]\), "+", 
        RowBox[{"6", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    " ", 
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(A\_1[T\_1]\^2\), " ", 
          \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"6", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    " ", 
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\^2\), 
          " ", \(A\&_\_1[T\_1]\)}], "-", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            SuperscriptBox[
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              "2"], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], "2"], \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Omega]\_\(1, m\)\%2\), " ", 
          \(A\_1[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(A\_2[T\_1]\^2\), " ", 
          \(A\&_\_1[T\_1]\)}], "-", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            RowBox[{
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_2[T\_1]\^2\), 
          " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_2[T\_1]\^2\), 
          " ", \(A\&_\_1[T\_1]\)}], "+", 
        FractionBox[
          RowBox[{
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_1\%x\), 
                            RowBox[{
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}]}], \(\[DifferentialD]x\)}]}],
                          HoldForm], ")"}], "2"], \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_2[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "+", 
        FractionBox[
          RowBox[{"2", " ", 
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_0\%x\), 
                            RowBox[{\(\[Integral]\_1\%x\), 
                              RowBox[{
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}]}], \(\[DifferentialD]x\), 
                                \(\[DifferentialD]x\)}]}]}],
                          HoldForm], ")"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_2[T\_1]\^2\), " ", \(A\&_\_1[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "-", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            SuperscriptBox[
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              "2"], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], " ", 
                    TagBox[
                      RowBox[{\(\[Integral]\_0\%x\), 
                        RowBox[{
                          SuperscriptBox[
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            "2"], \(\[DifferentialD]x\)}]}],
                      HoldForm]}], \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Omega]\_\(1, m\)\), " ", 
          \(\[Omega]\_\(2, n\)\), " ", \(A\_2[T\_1]\^2\), " ", 
          \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"4", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), 
          " ", \(A\&_\_2[T\_1]\)}], "-", 
        RowBox[{"4", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            RowBox[{
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\), 
          " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{"4", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\), 
          " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        FractionBox[
          RowBox[{"2", " ", 
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_1\%x\), 
                            RowBox[{
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}]}], \(\[DifferentialD]x\)}]}],
                          HoldForm], ")"}], "2"], \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "+", 
        FractionBox[
          RowBox[{"4", " ", 
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_0\%x\), 
                            RowBox[{\(\[Integral]\_1\%x\), 
                              RowBox[{
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}]}], \(\[DifferentialD]x\), 
                                \(\[DifferentialD]x\)}]}]}],
                          HoldForm], ")"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_2[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "+", 
        RowBox[{"2", " ", "I", " ", \(\[Omega]\_\(1, m\)\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "1", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eqMod2", "=", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"D", "[", 
              RowBox[{
                RowBox[{"D", "[", 
                  RowBox[{"TAL", ",", 
                    RowBox[{
                      SubsuperscriptBox[\(A\&_\), "2", "\[Prime]",
                        MultilineFunction->None], "[", \(T\_1\), "]"}]}], 
                  "]"}], ",", \(T\_1\)}], "]"}], "-", 
            \(D[TAL, A\&_\_2[T\_1]]\)}], "==", "0"}], "//.", 
        \(int[a_, b__] -> HoldForm[Integrate[a, b]]\)}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"4", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), 
          " ", \(A\&_\_1[T\_1]\)}], "-", 
        RowBox[{"4", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            RowBox[{
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\), 
          " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        RowBox[{"4", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\), 
          " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], "+", 
        FractionBox[
          RowBox[{"2", " ", 
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_1\%x\), 
                            RowBox[{
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}]}], \(\[DifferentialD]x\)}]}],
                          HoldForm], ")"}], "2"], \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "+", 
        FractionBox[
          RowBox[{"4", " ", 
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_0\%x\), 
                            RowBox[{\(\[Integral]\_1\%x\), 
                              RowBox[{
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}]}], \(\[DifferentialD]x\), 
                                \(\[DifferentialD]x\)}]}]}],
                          HoldForm], ")"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_1[T\_1]\), " ", \(A\_2[T\_1]\), " ", \(A\&_\_1[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "+", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(A\_1[T\_1]\^2\), " ", 
          \(A\&_\_2[T\_1]\)}], "-", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            RowBox[{
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              " ", 
                              RowBox[{
                                
                                SubsuperscriptBox["\[Phi]", "m", 
                                  "\[DoublePrime]",
                                  MultilineFunction->None], "[", "x", 
                                "]"}]}], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\^2\), 
          " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}], " ", 
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                        MultilineFunction->None], "[", "x", "]"}]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Delta]\_0\), " ", \(A\_1[T\_1]\^2\), 
          " ", \(A\&_\_2[T\_1]\)}], "+", 
        FractionBox[
          RowBox[{
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_1\%x\), 
                            RowBox[{
                              RowBox[{
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "m", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}], " ", 
                                RowBox[{
                                  
                                  SubsuperscriptBox["\[Phi]", "n", 
                                    "\[DoublePrime]",
                                    MultilineFunction->None], "[", "x", 
                                  "]"}]}], \(\[DifferentialD]x\)}]}],
                          HoldForm], ")"}], "2"], \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_1[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "+", 
        FractionBox[
          RowBox[{"2", " ", 
            RowBox[{"(", 
              TagBox[
                RowBox[{\(\[Integral]\_0\%1\), 
                  RowBox[{
                    RowBox[{
                      RowBox[{"(", 
                        TagBox[
                          RowBox[{\(\[Integral]\_0\%x\), 
                            RowBox[{\(\[Integral]\_1\%x\), 
                              RowBox[{
                                RowBox[{
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "m", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}], " ", 
                                  RowBox[{
                                    
                                    SubsuperscriptBox["\[Phi]", "n", 
                                      "\[DoublePrime]",
                                      MultilineFunction->None], "[", "x", 
                                    "]"}]}], \(\[DifferentialD]x\), 
                                \(\[DifferentialD]x\)}]}]}],
                          HoldForm], ")"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "m", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], " ", 
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}]}], 
                    \(\[DifferentialD]x\)}]}],
                HoldForm], ")"}], " ", \(\[Delta]\_0\%2\), " ", 
            \(A\_1[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\)}], 
          \(\[Beta]\_\[Gamma]\)], "-", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            SuperscriptBox[
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "m", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              "2"], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], " ", 
                    TagBox[
                      RowBox[{\(\[Integral]\_0\%x\), 
                        RowBox[{
                          SuperscriptBox[
                            RowBox[{
                              SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                MultilineFunction->None], "[", "x", "]"}], 
                            "2"], \(\[DifferentialD]x\)}]}],
                      HoldForm]}], \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Omega]\_\(1, m\)\), " ", 
          \(\[Omega]\_\(2, n\)\), " ", \(A\_1[T\_1]\^2\), " ", 
          \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{"6", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  RowBox[{
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"], 
                    " ", 
                    SuperscriptBox[
                      RowBox[{
                        SubsuperscriptBox["\[Phi]", "n", "\[DoublePrime]",
                          MultilineFunction->None], "[", "x", "]"}], "2"]}], 
                  \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(A\_2[T\_1]\^2\), " ", 
          \(A\&_\_2[T\_1]\)}], "-", 
        RowBox[{"2", " ", 
          RowBox[{"(", 
            TagBox[
              RowBox[{\(\[Integral]\_0\%1\), 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{"(", 
                      TagBox[
                        RowBox[{\(\[Integral]\_0\%x\), 
                          RowBox[{
                            SuperscriptBox[
                              RowBox[{
                                SubsuperscriptBox["\[Phi]", "n", "\[Prime]",
                                  MultilineFunction->None], "[", "x", "]"}], 
                              "2"], \(\[DifferentialD]x\)}]}],
                        HoldForm], ")"}], "2"], \(\[DifferentialD]x\)}]}],
              HoldForm], ")"}], " ", \(\[Omega]\_\(2, n\)\%2\), " ", 
          \(A\_2[T\_1]\^2\), " ", \(A\&_\_2[T\_1]\)}], "+", 
        RowBox[{"2", " ", "I", " ", \(\[Omega]\_\(2, n\)\), " ", 
          RowBox[{
            SubsuperscriptBox["A", "2", "\[Prime]",
              MultilineFunction->None], "[", \(T\_1\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Adding linear viscous damping to ",
  StyleBox["eqMod1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eqMod2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and performing integration by parts yields the same modulation equations \
as obtained by directly attacking the partial-differential system."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"Microsoft Windows 3.0",
ScreenRectangle->{{0, 1024}, {0, 712}},
WindowToolbars->{"RulerBar", "EditBar"},
WindowSize->{712, 567},
WindowMargins->{{0, Automatic}, {Automatic, 5}},
PrintingCopies->1,
PrintingStartingPageNumber->195,
PrintingPageRange->{Automatic, Automatic},
PageHeaders->{{Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"], Inherited, Cell[ 
        TextData[ {
          ValueBox[ "FileName"]}], "Header"]}, {Cell[ 
        TextData[ {
          ValueBox[ "FileName"]}], "Header"], Inherited, Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"]}},
PrintingOptions->{"FirstPageHeader"->False,
"FacingPages"->True},
Magnification->1
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 126, 5, 195, "Title"],
Cell[1860, 58, 983, 27, 109, "Text"],
Cell[2846, 87, 1503, 29, 185, "Text"],
Cell[4352, 118, 891, 14, 128, "Text"],
Cell[5246, 134, 247, 6, 40, "Text"],
Cell[5496, 142, 246, 9, 26, "Text"],
Cell[5745, 153, 705, 20, 54, "Text"],
Cell[6453, 175, 779, 17, 83, "Text"],
Cell[7235, 194, 638, 11, 83, "Text"],
Cell[7876, 207, 980, 15, 126, "Text"],
Cell[8859, 224, 757, 13, 97, "Text"],

Cell[CellGroupData[{
Cell[9641, 241, 79, 2, 46, "Section"],
Cell[9723, 245, 118, 3, 22, "Input"],
Cell[9844, 250, 112, 3, 22, "Input"],
Cell[9959, 255, 536, 21, 26, "Text"],
Cell[10498, 278, 456, 15, 22, "Input"],
Cell[10957, 295, 98, 2, 26, "Text"],
Cell[11058, 299, 117, 3, 22, "Input"],
Cell[11178, 304, 251, 8, 26, "Text"],
Cell[11432, 314, 324, 8, 34, "Input"],
Cell[11759, 324, 318, 8, 26, "Text"],
Cell[12080, 334, 267, 6, 22, "Input"],
Cell[12350, 342, 231, 7, 26, "Text"],
Cell[12584, 351, 499, 10, 72, "Input"],
Cell[13086, 363, 173, 5, 26, "Text"],
Cell[13262, 370, 904, 19, 119, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[14203, 394, 119, 2, 46, "Section"],
Cell[14325, 398, 617, 15, 69, "Text"],
Cell[14945, 415, 168, 3, 26, "Text"],
Cell[15116, 420, 215, 5, 34, "Input"],
Cell[15334, 427, 136, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[15495, 435, 165, 4, 22, "Input"],
Cell[15663, 441, 120, 3, 22, "Output"]
}, Open  ]],
Cell[15798, 447, 242, 6, 40, "Text"],
Cell[16043, 455, 292, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[16360, 470, 133, 3, 22, "Input"],
Cell[16496, 475, 171, 5, 26, "Output"]
}, Open  ]],
Cell[16682, 483, 343, 14, 26, "Text"],

Cell[CellGroupData[{
Cell[17050, 501, 131, 3, 22, "Input"],
Cell[17184, 506, 128, 3, 28, "Output"]
}, Open  ]],
Cell[17327, 512, 102, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[17454, 518, 138, 3, 22, "Input"],
Cell[17595, 523, 112, 3, 28, "Output"]
}, Open  ]],
Cell[17722, 529, 106, 2, 26, "Text"],
Cell[17831, 533, 622, 12, 69, "Text"],
Cell[18456, 547, 415, 12, 40, "Text"],
Cell[18874, 561, 1078, 20, 174, "Input"],
Cell[19955, 583, 688, 27, 40, "Text"],

Cell[CellGroupData[{
Cell[20668, 614, 144, 4, 22, "Input"],
Cell[20815, 620, 1225, 31, 61, "Output"]
}, Open  ]],
Cell[22055, 654, 339, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[22419, 669, 199, 5, 22, "Input"],
Cell[22621, 676, 262, 8, 22, "Output"]
}, Open  ]],
Cell[22898, 687, 581, 22, 40, "Text"],

Cell[CellGroupData[{
Cell[23504, 713, 192, 5, 22, "Input"],
Cell[23699, 720, 432, 13, 22, "Output"]
}, Open  ]],
Cell[24146, 736, 422, 15, 40, "Text"],

Cell[CellGroupData[{
Cell[24593, 755, 853, 21, 34, "Input"],
Cell[25449, 778, 103, 3, 22, "Output"]
}, Open  ]],
Cell[25567, 784, 730, 26, 40, "Text"],

Cell[CellGroupData[{
Cell[26322, 814, 127, 3, 22, "Input"],
Cell[26452, 819, 119, 3, 22, "Output"]
}, Open  ]],
Cell[26586, 825, 519, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[27130, 845, 274, 7, 34, "Input"],
Cell[27407, 854, 529, 14, 34, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[27973, 873, 112, 3, 22, "Input"],
Cell[28088, 878, 130, 3, 26, "Output"]
}, Open  ]],
Cell[28233, 884, 151, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[28409, 893, 94, 2, 43, "Subsection"],
Cell[28506, 897, 228, 6, 26, "Text"],
Cell[28737, 905, 600, 16, 22, "Input"],
Cell[29340, 923, 598, 16, 22, "Input"],
Cell[29941, 941, 409, 17, 26, "Text"],

Cell[CellGroupData[{
Cell[30375, 962, 150, 4, 22, "Input"],
Cell[30528, 968, 3571, 88, 85, "Output"]
}, Open  ]],
Cell[34114, 1059, 353, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[34492, 1074, 215, 6, 22, "Input"],
Cell[34710, 1082, 479, 14, 22, "Output"]
}, Open  ]],
Cell[35204, 1099, 369, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[35598, 1116, 163, 5, 22, "Input"],
Cell[35764, 1123, 1286, 35, 22, "Output"]
}, Open  ]],
Cell[37065, 1161, 498, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[37588, 1181, 1439, 32, 47, "Input"],
Cell[39030, 1215, 483, 14, 22, "Output"]
}, Open  ]],
Cell[39528, 1232, 651, 23, 40, "Text"],
Cell[40182, 1257, 520, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[40727, 1277, 275, 7, 34, "Input"],
Cell[41005, 1286, 136, 3, 34, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[41190, 1295, 90, 2, 43, "Subsection"],
Cell[41283, 1299, 195, 5, 26, "Text"],
Cell[41481, 1306, 408, 12, 22, "Input"],
Cell[41892, 1320, 541, 15, 22, "Input"],
Cell[42436, 1337, 505, 18, 40, "Text"],

Cell[CellGroupData[{
Cell[42966, 1359, 150, 4, 22, "Input"],
Cell[43119, 1365, 2608, 68, 73, "Output"]
}, Open  ]],
Cell[45742, 1436, 343, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[46110, 1451, 215, 6, 22, "Input"],
Cell[46328, 1459, 308, 10, 22, "Output"]
}, Open  ]],
Cell[46651, 1472, 369, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[47045, 1489, 163, 5, 22, "Input"],
Cell[47211, 1496, 1101, 31, 22, "Output"]
}, Open  ]],
Cell[48327, 1530, 485, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[48837, 1550, 1253, 30, 47, "Input"],
Cell[50093, 1582, 656, 19, 22, "Output"]
}, Open  ]],
Cell[50764, 1604, 651, 23, 40, "Text"],
Cell[51418, 1629, 520, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[51963, 1649, 275, 7, 34, "Input"],
Cell[52241, 1658, 136, 3, 34, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[52438, 1668, 89, 2, 46, "Section"],

Cell[CellGroupData[{
Cell[52552, 1674, 149, 4, 43, "Subsection"],
Cell[52704, 1680, 748, 18, 83, "Text"],

Cell[CellGroupData[{
Cell[53477, 1702, 538, 11, 48, "Input"],
Cell[54018, 1715, 1867, 52, 47, "Output"]
}, Open  ]],
Cell[55900, 1770, 98, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[56023, 1776, 206, 4, 22, "Input"],
Cell[56232, 1782, 593, 18, 22, "Output"]
}, Open  ]],
Cell[56840, 1803, 631, 11, 83, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[57508, 1819, 114, 2, 43, "Subsection"],
Cell[57625, 1823, 432, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[58082, 1840, 179, 5, 22, "Input"],
Cell[58264, 1847, 148, 4, 22, "Output"]
}, Open  ]],
Cell[58427, 1854, 1020, 33, 69, "Text"],
Cell[59450, 1889, 483, 9, 98, "Input"],
Cell[59936, 1900, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[60191, 1912, 219, 5, 34, "Input"],
Cell[60413, 1919, 8462, 192, 66, "Output"]
}, Open  ]],
Cell[68890, 2114, 458, 11, 54, "Text"],
Cell[69351, 2127, 302, 7, 35, "Input"],
Cell[69656, 2136, 303, 11, 26, "Text"],
Cell[69962, 2149, 122, 3, 22, "Input"],
Cell[70087, 2154, 251, 7, 26, "Text"],
Cell[70341, 2163, 190, 4, 25, "Text"],
Cell[70534, 2169, 195, 9, 26, "Text"],
Cell[70732, 2180, 504, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[71261, 2204, 397, 8, 60, "Input"],
Cell[71661, 2214, 28431, 645, 541, "Output"]
}, Open  ]],
Cell[100107, 2862, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[100362, 2874, 113, 3, 22, "Input"],
Cell[100478, 2879, 663, 18, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[101178, 2902, 174, 4, 34, "Subsubsection"],
Cell[101355, 2908, 462, 14, 40, "Text"],
Cell[101820, 2924, 242, 6, 22, "Input"],
Cell[102065, 2932, 94, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[102184, 2938, 206, 5, 22, "Input"],
Cell[102393, 2945, 485, 10, 28, "Output"]
}, Open  ]],
Cell[102893, 2958, 191, 6, 22, "Input"],
Cell[103087, 2966, 346, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[103458, 2982, 212, 6, 34, "Input"],
Cell[103673, 2990, 7921, 171, 172, "Output"]
}, Open  ]],
Cell[111609, 3164, 836, 25, 69, "Text"],

Cell[CellGroupData[{
Cell[112470, 3193, 265, 6, 47, "Input"],
Cell[112738, 3201, 11467, 241, 298, "Output"]
}, Open  ]],
Cell[124220, 3445, 210, 5, 40, "Text"],
Cell[124433, 3452, 814, 19, 34, "Input"],
Cell[125250, 3473, 270, 6, 40, "Text"],

Cell[CellGroupData[{
Cell[125545, 3483, 238, 6, 22, "Input"],
Cell[125786, 3491, 282, 6, 22, "Output"]
}, Open  ]],
Cell[126083, 3500, 167, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[126275, 3509, 213, 5, 34, "Input"],
Cell[126491, 3516, 1406, 24, 66, "Output"]
}, Open  ]],
Cell[127912, 3543, 213, 5, 40, "Text"],

Cell[CellGroupData[{
Cell[128150, 3552, 276, 6, 34, "Input"],
Cell[128429, 3560, 374, 8, 38, "Output"]
}, Open  ]],
Cell[128818, 3571, 104, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[128947, 3577, 247, 6, 22, "Input"],
Cell[129197, 3585, 320, 6, 22, "Output"]
}, Open  ]],
Cell[129532, 3594, 202, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[129759, 3606, 358, 8, 61, "Input"],
Cell[130120, 3616, 1555, 33, 76, "Output"]
}, Open  ]],
Cell[131690, 3652, 68, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[131783, 3658, 239, 5, 34, "Input"],
Cell[132025, 3665, 7257, 164, 61, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[139319, 3834, 239, 5, 34, "Input"],
Cell[139561, 3841, 5691, 130, 61, "Output"]
}, Open  ]],
Cell[145267, 3974, 211, 6, 42, "Text"],
Cell[145481, 3982, 191, 5, 26, "Text"],
Cell[145675, 3989, 232, 5, 32, "Input"],
Cell[145910, 3996, 180, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[146115, 4008, 191, 5, 22, "Input"],
Cell[146309, 4015, 1972, 41, 59, "Output"]
}, Open  ]],
Cell[148296, 4059, 455, 14, 40, "Text"],
Cell[148754, 4075, 123, 3, 22, "Input"],
Cell[148880, 4080, 268, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[149173, 4093, 812, 20, 34, "Input"],
Cell[149988, 4115, 1198, 26, 58, "Output"]
}, Open  ]],
Cell[151201, 4144, 263, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[151489, 4157, 315, 7, 34, "Input"],
Cell[151807, 4166, 1383, 29, 104, "Output"]
}, Open  ]],
Cell[153205, 4198, 218, 8, 26, "Text"],
Cell[153426, 4208, 415, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[153866, 4223, 147, 4, 22, "Input"],
Cell[154016, 4229, 469, 10, 33, "Output"]
}, Open  ]],
Cell[154500, 4242, 119, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[154644, 4248, 158, 4, 22, "Input"],
Cell[154805, 4254, 121, 3, 22, "Output"]
}, Open  ]],
Cell[154941, 4260, 265, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[155231, 4274, 180, 4, 22, "Input"],
Cell[155414, 4280, 284, 6, 22, "Output"]
}, Open  ]],
Cell[155713, 4289, 246, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[155984, 4303, 192, 5, 22, "Input"],
Cell[156179, 4310, 309, 7, 28, "Output"]
}, Open  ]],
Cell[156503, 4320, 260, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[156788, 4335, 279, 7, 47, "Input"],
Cell[157070, 4344, 2008, 41, 75, "Output"]
}, Open  ]],
Cell[159093, 4388, 268, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[159386, 4401, 985, 24, 47, "Input"],
Cell[160374, 4427, 706, 16, 33, "Output"]
}, Open  ]],
Cell[161095, 4446, 263, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[161383, 4459, 354, 7, 60, "Input"],
Cell[161740, 4468, 1132, 23, 67, "Output"]
}, Open  ]],
Cell[162887, 4494, 236, 8, 26, "Text"],
Cell[163126, 4504, 238, 7, 42, "Text"],
Cell[163367, 4513, 389, 8, 54, "Text"],
Cell[163759, 4523, 401, 9, 60, "Input"],
Cell[164163, 4534, 478, 18, 40, "Text"],
Cell[164644, 4554, 204, 5, 22, "Input"],
Cell[164851, 4561, 397, 12, 40, "Text"],

Cell[CellGroupData[{
Cell[165273, 4577, 242, 7, 34, "Input"],
Cell[165518, 4586, 473, 10, 33, "Output"]
}, Open  ]],
Cell[166006, 4599, 304, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[166335, 4614, 157, 4, 22, "Input"],
Cell[166495, 4620, 121, 3, 22, "Output"]
}, Open  ]],
Cell[166631, 4626, 605, 18, 54, "Text"],
Cell[167239, 4646, 113, 3, 22, "Input"],

Cell[CellGroupData[{
Cell[167377, 4653, 210, 5, 22, "Input"],
Cell[167590, 4660, 327, 8, 28, "Output"]
}, Open  ]],
Cell[167932, 4671, 283, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[168240, 4685, 111, 3, 22, "Input"],
Cell[168354, 4690, 191, 5, 28, "Output"]
}, Open  ]],
Cell[168560, 4698, 286, 8, 40, "Text"],

Cell[CellGroupData[{
Cell[168871, 4710, 532, 11, 85, "Input"],
Cell[169406, 4723, 7826, 228, 59, "Output"]
}, Open  ]],
Cell[177247, 4954, 139, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[177411, 4962, 242, 6, 47, "Input"],
Cell[177656, 4970, 105, 3, 22, "Output"]
}, Open  ]],
Cell[177776, 4976, 142, 4, 22, "Input"],

Cell[CellGroupData[{
Cell[177943, 4984, 187, 5, 22, "Input"],
Cell[178133, 4991, 160, 4, 22, "Output"]
}, Open  ]],
Cell[178308, 4998, 479, 12, 54, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[178824, 5015, 289, 11, 34, "Subsubsection"],
Cell[179116, 5028, 1308, 37, 97, "Text"],
Cell[180427, 5067, 2474, 47, 406, "Input"],
Cell[182904, 5116, 2800, 55, 392, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[185741, 5176, 121, 2, 34, "Subsubsection"],
Cell[185865, 5180, 459, 14, 40, "Text"],
Cell[186327, 5196, 242, 6, 22, "Input"],
Cell[186572, 5204, 94, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[186691, 5210, 206, 5, 22, "Input"],
Cell[186900, 5217, 453, 10, 28, "Output"]
}, Open  ]],
Cell[187368, 5230, 191, 6, 22, "Input"],
Cell[187562, 5238, 346, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[187933, 5254, 212, 6, 34, "Input"],
Cell[188148, 5262, 7319, 158, 144, "Output"]
}, Open  ]],
Cell[195482, 5423, 668, 23, 40, "Text"],

Cell[CellGroupData[{
Cell[196175, 5450, 265, 6, 47, "Input"],
Cell[196443, 5458, 10370, 218, 298, "Output"]
}, Open  ]],
Cell[206828, 5679, 262, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[207115, 5691, 358, 8, 61, "Input"],
Cell[207476, 5701, 1310, 29, 76, "Output"]
}, Open  ]],
Cell[208801, 5733, 68, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[208894, 5739, 239, 5, 34, "Input"],
Cell[209136, 5746, 7257, 164, 61, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[216430, 5915, 239, 5, 34, "Input"],
Cell[216672, 5922, 5691, 130, 61, "Output"]
}, Open  ]],
Cell[222378, 6055, 211, 6, 42, "Text"],
Cell[222592, 6063, 454, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[223071, 6083, 96, 3, 22, "Input"],
Cell[223170, 6088, 2157, 45, 111, "Output"]
}, Open  ]],
Cell[225342, 6136, 238, 7, 42, "Text"],
Cell[225583, 6145, 483, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[226091, 6165, 100, 3, 22, "Input"],
Cell[226194, 6170, 3362, 64, 180, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[229605, 6240, 121, 2, 34, "Subsubsection"],
Cell[229729, 6244, 482, 15, 40, "Text"],
Cell[230214, 6261, 263, 7, 22, "Input"],
Cell[230480, 6270, 94, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[230599, 6276, 206, 5, 22, "Input"],
Cell[230808, 6283, 532, 12, 28, "Output"]
}, Open  ]],
Cell[231355, 6298, 191, 6, 22, "Input"],
Cell[231549, 6306, 346, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[231920, 6322, 212, 6, 34, "Input"],
Cell[232135, 6330, 7591, 164, 172, "Output"]
}, Open  ]],
Cell[239741, 6497, 308, 9, 40, "Text"],

Cell[CellGroupData[{
Cell[240074, 6510, 265, 6, 47, "Input"],
Cell[240342, 6518, 10888, 229, 298, "Output"]
}, Open  ]],
Cell[251245, 6750, 262, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[251532, 6762, 358, 8, 61, "Input"],
Cell[251893, 6772, 1410, 31, 75, "Output"]
}, Open  ]],
Cell[253318, 6806, 68, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[253411, 6812, 239, 5, 34, "Input"],
Cell[253653, 6819, 7257, 164, 61, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[260947, 6988, 239, 5, 34, "Input"],
Cell[261189, 6995, 5691, 130, 61, "Output"]
}, Open  ]],
Cell[266895, 7128, 211, 6, 42, "Text"],
Cell[267109, 7136, 421, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[267555, 7156, 96, 3, 22, "Input"],
Cell[267654, 7161, 2737, 56, 144, "Output"]
}, Open  ]],
Cell[270406, 7220, 238, 7, 42, "Text"],
Cell[270647, 7229, 450, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[271122, 7249, 100, 3, 22, "Input"],
Cell[271225, 7254, 3121, 59, 156, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[274407, 7320, 115, 2, 43, "Subsection"],
Cell[274525, 7324, 549, 19, 40, "Text"],
Cell[275077, 7345, 191, 6, 37, "Input"],
Cell[275271, 7353, 369, 11, 40, "Text"],
Cell[275643, 7366, 141, 4, 22, "Input"],
Cell[275787, 7372, 907, 33, 55, "Text"],

Cell[CellGroupData[{
Cell[276719, 7409, 268, 6, 47, "Input"],
Cell[276990, 7417, 2840, 67, 89, "Output"]
}, Open  ]],
Cell[279845, 7487, 169, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[280039, 7496, 606, 15, 22, "Input"],
Cell[280648, 7513, 975, 23, 34, "Output"]
}, Open  ]],
Cell[281638, 7539, 280, 8, 40, "Text"],
Cell[281921, 7549, 1232, 29, 34, "Input"],
Cell[283156, 7580, 1965, 43, 63, "Input"],
Cell[285124, 7625, 185, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[285334, 7637, 215, 6, 34, "Input"],
Cell[285552, 7645, 724, 16, 22, "Output"]
}, Open  ]],
Cell[286291, 7664, 242, 10, 26, "Text"],
Cell[286536, 7676, 161, 5, 26, "Text"],
Cell[286700, 7683, 187, 5, 39, "Input"],
Cell[286890, 7690, 669, 23, 40, "Text"],

Cell[CellGroupData[{
Cell[287584, 7717, 333, 7, 60, "Input"],
Cell[287920, 7726, 1518, 37, 46, "Output"]
}, Open  ]],
Cell[289453, 7766, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[289708, 7778, 150, 4, 22, "Input"],
Cell[289861, 7784, 1635, 40, 47, "Output"]
}, Open  ]],
Cell[291511, 7827, 538, 18, 40, "Text"],
Cell[292052, 7847, 258, 7, 22, "Input"],
Cell[292313, 7856, 114, 2, 26, "Text"],
Cell[292430, 7860, 617, 13, 79, "Input"],
Cell[293050, 7875, 190, 8, 26, "Text"],
Cell[293243, 7885, 95, 3, 22, "Input"],

Cell[CellGroupData[{
Cell[293363, 7892, 284, 6, 47, "Input"],
Cell[293650, 7900, 2003, 43, 84, "Output"]
}, Open  ]],
Cell[295668, 7946, 376, 16, 26, "Text"],
Cell[296047, 7964, 158, 4, 22, "Input"],

Cell[CellGroupData[{
Cell[296230, 7972, 174, 4, 34, "Subsubsection"],
Cell[296407, 7978, 242, 6, 22, "Input"],

Cell[CellGroupData[{
Cell[296674, 7988, 206, 5, 22, "Input"],
Cell[296883, 7995, 485, 10, 28, "Output"]
}, Open  ]],
Cell[297383, 8008, 191, 6, 22, "Input"],
Cell[297577, 8016, 345, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[297947, 8032, 236, 6, 34, "Input"],
Cell[298186, 8040, 2473, 49, 90, "Output"]
}, Open  ]],
Cell[300674, 8092, 123, 2, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[300834, 8099, 175, 4, 34, "Subsubsection"],
Cell[301012, 8105, 242, 6, 22, "Input"],

Cell[CellGroupData[{
Cell[301279, 8115, 206, 5, 22, "Input"],
Cell[301488, 8122, 453, 10, 28, "Output"]
}, Open  ]],
Cell[301956, 8135, 191, 6, 22, "Input"],
Cell[302150, 8143, 345, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[302520, 8159, 236, 6, 34, "Input"],
Cell[302759, 8167, 2226, 45, 77, "Output"]
}, Open  ]],
Cell[305000, 8215, 123, 2, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[305160, 8222, 175, 4, 34, "Subsubsection"],
Cell[305338, 8228, 263, 7, 22, "Input"],

Cell[CellGroupData[{
Cell[305626, 8239, 206, 5, 22, "Input"],
Cell[305835, 8246, 532, 12, 28, "Output"]
}, Open  ]],
Cell[306382, 8261, 191, 6, 22, "Input"],
Cell[306576, 8269, 345, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[306946, 8285, 236, 6, 34, "Input"],
Cell[307185, 8293, 2327, 47, 77, "Output"]
}, Open  ]],
Cell[309527, 8343, 123, 2, 26, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[309699, 8351, 109, 2, 43, "Subsection"],
Cell[309811, 8355, 260, 6, 40, "Text"],
Cell[310074, 8363, 648, 14, 65, "Input"],
Cell[310725, 8379, 515, 21, 40, "Text"],

Cell[CellGroupData[{
Cell[311265, 8404, 273, 6, 34, "Input"],
Cell[311541, 8412, 2126, 54, 63, "Output"]
}, Open  ]],
Cell[313682, 8469, 82, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[313789, 8475, 302, 8, 22, "Input"],
Cell[314094, 8485, 378, 8, 22, "Output"]
}, Open  ]],
Cell[314487, 8496, 929, 34, 54, "Text"],
Cell[315419, 8532, 230, 5, 34, "Input"],

Cell[CellGroupData[{
Cell[315674, 8541, 174, 4, 34, "Subsubsection"],

Cell[CellGroupData[{
Cell[315873, 8549, 263, 6, 22, "Input"],
Cell[316139, 8557, 219, 5, 22, "Output"]
}, Open  ]],
Cell[316373, 8565, 314, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[316712, 8581, 281, 8, 47, "Input"],
Cell[316996, 8591, 7730, 179, 262, "Output"]
}, Open  ]],
Cell[324741, 8773, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[324996, 8785, 522, 14, 22, "Input"],
Cell[325521, 8801, 4488, 103, 135, "Output"]
}, Open  ]],
Cell[330024, 8907, 66, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[330115, 8913, 522, 14, 22, "Input"],
Cell[330640, 8929, 4011, 93, 135, "Output"]
}, Open  ]],
Cell[334666, 9025, 431, 13, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[335134, 9043, 121, 2, 34, "Subsubsection"],

Cell[CellGroupData[{
Cell[335280, 9049, 300, 6, 22, "Input"],
Cell[335583, 9057, 261, 6, 22, "Output"]
}, Open  ]],
Cell[335859, 9066, 314, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[336198, 9082, 281, 8, 47, "Input"],
Cell[336482, 9092, 6443, 149, 234, "Output"]
}, Open  ]],
Cell[342940, 9244, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[343195, 9256, 522, 14, 22, "Input"],
Cell[343720, 9272, 3390, 78, 107, "Output"]
}, Open  ]],
Cell[347125, 9353, 66, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[347216, 9359, 522, 14, 22, "Input"],
Cell[347741, 9375, 3803, 87, 135, "Output"]
}, Open  ]],
Cell[351559, 9465, 431, 13, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[352027, 9483, 121, 2, 34, "Subsubsection"],

Cell[CellGroupData[{
Cell[352173, 9489, 326, 7, 22, "Input"],
Cell[352502, 9498, 258, 6, 22, "Output"]
}, Open  ]],
Cell[352775, 9507, 314, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[353114, 9523, 281, 8, 47, "Input"],
Cell[353398, 9533, 6807, 156, 206, "Output"]
}, Open  ]],
Cell[360220, 9692, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[360475, 9704, 522, 14, 22, "Input"],
Cell[361000, 9720, 3995, 91, 107, "Output"]
}, Open  ]],
Cell[365010, 9814, 66, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[365101, 9820, 522, 14, 22, "Input"],
Cell[365626, 9836, 3955, 90, 135, "Output"]
}, Open  ]],
Cell[369596, 9929, 431, 13, 40, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[370088, 9949, 85, 2, 46, "Section"],

Cell[CellGroupData[{
Cell[370198, 9955, 149, 4, 43, "Subsection"],
Cell[370350, 9961, 846, 22, 83, "Text"],
Cell[371199, 9985, 3892, 74, 224, "Input"],
Cell[375094, 10061, 109, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[375228, 10067, 250, 5, 34, "Input"],
Cell[375481, 10074, 791, 25, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[376309, 10104, 109, 3, 22, "Input"],
Cell[376421, 10109, 1477, 46, 33, "Output"]
}, Open  ]],
Cell[377913, 10158, 344, 10, 40, "Text"],

Cell[CellGroupData[{
Cell[378282, 10172, 458, 10, 61, "Input"],
Cell[378743, 10184, 1502, 43, 33, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[380282, 10232, 397, 9, 47, "Input"],
Cell[380682, 10243, 756, 22, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[381475, 10270, 447, 11, 33, "Input"],
Cell[381925, 10283, 2068, 52, 50, "Output"]
}, Open  ]],
Cell[384008, 10338, 437, 14, 40, "Text"],
Cell[384448, 10354, 182, 8, 26, "Text"],
Cell[384633, 10364, 1600, 33, 155, "Input"],
Cell[386236, 10399, 362, 13, 26, "Text"],
Cell[386601, 10414, 479, 10, 53, "Input"],
Cell[387083, 10426, 490, 19, 26, "Text"],
Cell[387576, 10447, 150, 3, 24, "Text"],
Cell[387729, 10452, 734, 26, 40, "Text"],
Cell[388466, 10480, 167, 5, 22, "Input"],
Cell[388636, 10487, 632, 18, 54, "Text"],
Cell[389271, 10507, 143, 4, 26, "Text"],
Cell[389417, 10513, 728, 16, 98, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[390182, 10534, 114, 2, 43, "Subsection"],
Cell[390299, 10538, 432, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[390756, 10555, 278, 7, 34, "Input"],
Cell[391037, 10564, 223, 5, 22, "Output"]
}, Open  ]],
Cell[391275, 10572, 1375, 48, 69, "Text"],

Cell[CellGroupData[{
Cell[392675, 10624, 599, 11, 136, "Input"],
Cell[393277, 10637, 117, 3, 22, "Output"]
}, Open  ]],
Cell[393409, 10643, 230, 8, 26, "Text"],
Cell[393642, 10653, 201, 5, 22, "Input"],

Cell[CellGroupData[{
Cell[393868, 10662, 109, 3, 22, "Input"],
Cell[393980, 10667, 4551, 122, 47, "Output"]
}, Open  ]],
Cell[398546, 10792, 434, 11, 54, "Text"],
Cell[398983, 10805, 529, 11, 35, "Input"],
Cell[399515, 10818, 251, 7, 26, "Text"],
Cell[399769, 10827, 190, 4, 25, "Text"],
Cell[399962, 10833, 195, 9, 26, "Text"],
Cell[400160, 10844, 602, 24, 26, "Text"],
Cell[400765, 10870, 95, 3, 22, "Input"],
Cell[400863, 10875, 379, 8, 60, "Input"],
Cell[401245, 10885, 379, 8, 60, "Input"],
Cell[401627, 10895, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[401882, 10907, 114, 3, 22, "Input"],
Cell[401999, 10912, 1635, 46, 35, "Output"]
}, Open  ]],
Cell[403649, 10961, 480, 15, 40, "Text"],
Cell[404132, 10978, 136, 3, 22, "Input"],
Cell[404271, 10983, 219, 5, 22, "Input"],
Cell[404493, 10990, 94, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[404612, 10996, 206, 5, 22, "Input"],
Cell[404821, 11003, 347, 7, 22, "Output"]
}, Open  ]],
Cell[405183, 11013, 191, 6, 22, "Input"],
Cell[405377, 11021, 445, 16, 40, "Text"],
Cell[405825, 11039, 207, 5, 22, "Input"],
Cell[406035, 11046, 512, 19, 40, "Text"],
Cell[406550, 11067, 186, 5, 22, "Input"],
Cell[406739, 11074, 186, 5, 22, "Input"],
Cell[406928, 11081, 313, 10, 26, "Text"],
Cell[407244, 11093, 210, 5, 34, "Input"],
Cell[407457, 11100, 271, 12, 26, "Text"],
Cell[407731, 11114, 171, 4, 22, "Input"],
Cell[407905, 11120, 210, 5, 40, "Text"],
Cell[408118, 11127, 846, 19, 36, "Input"],
Cell[408967, 11148, 270, 6, 40, "Text"],

Cell[CellGroupData[{
Cell[409262, 11158, 248, 6, 22, "Input"],
Cell[409513, 11166, 310, 6, 22, "Output"]
}, Open  ]],
Cell[409838, 11175, 167, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[410030, 11184, 213, 5, 34, "Input"],
Cell[410246, 11191, 1649, 29, 64, "Output"]
}, Open  ]],
Cell[411910, 11223, 213, 5, 40, "Text"],

Cell[CellGroupData[{
Cell[412148, 11232, 281, 6, 34, "Input"],
Cell[412432, 11240, 535, 9, 46, "Output"]
}, Open  ]],
Cell[412982, 11252, 104, 2, 26, "Text"],
Cell[413089, 11256, 207, 5, 22, "Input"],
Cell[413299, 11263, 97, 3, 22, "Input"],

Cell[CellGroupData[{
Cell[413421, 11270, 223, 6, 22, "Input"],
Cell[413647, 11278, 233, 5, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[413917, 11288, 223, 6, 22, "Input"],
Cell[414143, 11296, 233, 5, 22, "Output"]
}, Open  ]],
Cell[414391, 11304, 202, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[414618, 11316, 355, 8, 89, "Input"],
Cell[414976, 11326, 1373, 31, 71, "Output"]
}, Open  ]],
Cell[416364, 11360, 68, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[416457, 11366, 192, 4, 22, "Input"],
Cell[416652, 11372, 34207, 649, 497, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[450896, 12026, 192, 4, 22, "Input"],
Cell[451091, 12032, 23681, 451, 344, "Output"]
}, Open  ]],
Cell[474787, 12486, 211, 6, 42, "Text"],
Cell[475001, 12494, 454, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[475480, 12514, 96, 3, 22, "Input"],
Cell[475579, 12519, 2354, 47, 116, "Output"]
}, Open  ]],
Cell[477948, 12569, 238, 7, 42, "Text"],
Cell[478189, 12578, 483, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[478697, 12598, 100, 3, 22, "Input"],
Cell[478800, 12603, 3417, 64, 159, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[482266, 12673, 115, 2, 43, "Subsection"],
Cell[482384, 12677, 700, 25, 40, "Text"],
Cell[483087, 12704, 201, 4, 22, "Input"],
Cell[483291, 12710, 122, 3, 22, "Input"],
Cell[483416, 12715, 1534, 55, 83, "Text"],

Cell[CellGroupData[{
Cell[484975, 12774, 398, 8, 72, "Input"],
Cell[485376, 12784, 58447, 1162, 991, "Output"]
}, Open  ]],
Cell[543838, 13949, 243, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[544106, 13961, 903, 20, 49, "Input"],
Cell[545012, 13983, 1149, 32, 34, "Output"]
}, Open  ]],
Cell[546176, 14018, 248, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[546449, 14030, 129, 3, 22, "Input"],
Cell[546581, 14035, 387, 10, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[547005, 14050, 132, 3, 22, "Input"],
Cell[547140, 14055, 676, 17, 22, "Output"]
}, Open  ]],
Cell[547831, 14075, 206, 5, 22, "Input"],
Cell[548040, 14082, 309, 12, 26, "Text"],
Cell[548352, 14096, 169, 5, 22, "Input"],
Cell[548524, 14103, 188, 7, 26, "Text"],
Cell[548715, 14112, 192, 5, 22, "Input"],
Cell[548910, 14119, 471, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[549406, 14143, 264, 7, 47, "Input"],
Cell[549673, 14152, 3766, 83, 59, "Output"]
}, Open  ]],
Cell[553454, 14238, 161, 5, 26, "Text"],
Cell[553618, 14245, 187, 5, 39, "Input"],
Cell[553808, 14252, 763, 27, 40, "Text"],
Cell[554574, 14281, 373, 8, 60, "Input"],
Cell[554950, 14291, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[555205, 14303, 195, 5, 22, "Input"],
Cell[555403, 14310, 6994, 158, 150, "Output"]
}, Open  ]],
Cell[562412, 14471, 195, 8, 26, "Text"],
Cell[562610, 14481, 472, 12, 35, "Input"],
Cell[563085, 14495, 279, 12, 26, "Text"],
Cell[563367, 14509, 447, 9, 61, "Input"],
Cell[563817, 14520, 488, 15, 40, "Text"],
Cell[564308, 14537, 136, 3, 22, "Input"],
Cell[564447, 14542, 219, 5, 22, "Input"],

Cell[CellGroupData[{
Cell[564691, 14551, 206, 5, 22, "Input"],
Cell[564900, 14558, 347, 7, 22, "Output"]
}, Open  ]],
Cell[565262, 14568, 191, 6, 22, "Input"],
Cell[565456, 14576, 345, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[565826, 14592, 241, 6, 47, "Input"],
Cell[566070, 14600, 2639, 51, 94, "Output"]
}, Open  ]],
Cell[568724, 14654, 68, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[568817, 14660, 343, 7, 35, "Input"],
Cell[569163, 14669, 40542, 832, 592, "Output"]
}, Open  ]],
Cell[609720, 15504, 123, 2, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[609880, 15511, 109, 2, 43, "Subsection"],
Cell[609992, 15515, 313, 7, 54, "Text"],
Cell[610308, 15524, 2785, 47, 268, "Input"],
Cell[613096, 15573, 617, 25, 40, "Text"],
Cell[613716, 15600, 288, 7, 34, "Input"],
Cell[614007, 15609, 82, 2, 26, "Text"],
Cell[614092, 15613, 526, 11, 35, "Input"],
Cell[614621, 15626, 73, 2, 26, "Text"],
Cell[614697, 15630, 415, 9, 47, "Input"],
Cell[615115, 15641, 802, 32, 40, "Text"],

Cell[CellGroupData[{
Cell[615942, 15677, 444, 8, 98, "Input"],
Cell[616389, 15687, 117, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[616543, 15695, 238, 5, 22, "Input"],
Cell[616784, 15702, 205, 5, 22, "Output"]
}, Open  ]],
Cell[617004, 15710, 714, 18, 36, "Input"],
Cell[617721, 15730, 412, 16, 26, "Text"],
Cell[618136, 15748, 284, 8, 47, "Input"],
Cell[618423, 15758, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[618678, 15770, 624, 16, 35, "Input"],
Cell[619305, 15788, 17481, 368, 389, "Output"]
}, Open  ]],
Cell[636801, 16159, 66, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[636892, 16165, 624, 16, 35, "Input"],
Cell[637519, 16183, 16594, 349, 396, "Output"]
}, Open  ]],
Cell[654128, 16535, 431, 13, 40, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

